#include "stdafx.h"
#include "ttd.h"
#include "station.h"
#include "city.h"

#define USE_TABLE(x)  { assert(index < lengthof(x)); str = x[index]; break; }

static byte *StationGetSpecialString(byte *buff);
static byte *GetSpecialCityNameString(byte *buff, int ind);
static byte *GetSpecialPlayerNameString(byte *buff, int ind);

static byte *DecodeString(byte *buff, const byte *str);

static byte **_langpack_offs;
static uint _langtab_num[32]; // Offset into langpack offs
static uint _langtab_start[32]; // Offset into langpack offs

byte * const _special_string_table[1] = {
	_station_coverage_text
};

byte *_str_300c_string;

typedef byte *PlayerNameGeneratorProc(byte *buffr);

typedef struct {
	uint16 rate;
	char separator;
	byte flags;
	char pre[4];
	char post[4];
} CurrencySpec;

enum {
	CF_TOEURO_2002 = 1,
	CF_ISEURO = 2,
};

static const CurrencySpec _currency_specs[] = {
	{1,		',', 0,								"\xA3", ""},	// british pounds
	{2,		',', 0,								"$", ""},				// us dollars
	{10,	'.', CF_TOEURO_2002,	"FF ", ""},			// french francs
	{4,		'.', CF_TOEURO_2002,	"DM ", ""},// deutsche mark
	{200, ',', 0,								"\xA5", ""}, // yen
	{200, '.', CF_TOEURO_2002,	"Pt", ""},		// spanish pesetas
	{376, ',', CF_TOEURO_2002,	"Ft", ""},
	{6,   ' ', 0,								"", " zl"},
	{19,  ',', CF_TOEURO_2002,	"ATS ", ""},
	{57,  ',', CF_TOEURO_2002,	"BEF ", ""},
	{10,  '.', 0,								"", " kr"},
	{8,   ',', CF_TOEURO_2002,	"FIM ", ""},
	{480, ',', CF_TOEURO_2002,	"GRD ", ""},
	{2,   ',', 0,								"CHF ", ""},
	{3,   ',', CF_TOEURO_2002,	"NLG ", ""},
	{2730,',', CF_TOEURO_2002,	"ITL ", ""},
	{13,  '.', 0,								"", " kr"},
	{5,   ' ', 0,								"", " rur"},
	{2,   ',', CF_ISEURO,				"EUR ", ""},
};

const uint16 _currency_string_list[] = {
	STR_CURR_POUNDS,
	STR_CURR_DOLLARS,
	STR_CURR_FF,
	STR_CURR_DM,
	STR_CURR_YEN,
	STR_CURR_PT,
	STR_CURR_FT,
	STR_CURR_ZL,
	STR_CURR_ATS,
	STR_CURR_BEF,
	STR_CURR_DKK,
	STR_CURR_FIM,
	STR_CURR_GRD,
	STR_CURR_CHF,
	STR_CURR_NLG,
	STR_CURR_ITL,
	STR_CURR_SEK,
	STR_CURR_RUR,
	STR_CURR_EUR,
	INVALID_STRING_ID
};

// get a mask of the allowed currencies depending on the year
uint GetMaskOfAllowedCurrencies()
{
	int i;
	uint mask = 0;
	for(i=0; i!=lengthof(_currency_specs); i++) {
		byte flags = _currency_specs[i].flags;
		if (_cur_year >= (2002-1920) && (flags & CF_TOEURO_2002)) continue;
		if (_cur_year < (2000-1920) && (flags & CF_ISEURO)) continue;
		mask |= (1 << i);
	}
	return mask;
}

static byte *str_cat(byte *dst, const byte *src)
{
	while ( (*dst++ = *src++) != 0) {}
	return dst - 1;
}

static byte *GetStringPtr(uint16 string)
{
	return _langpack_offs[_langtab_start[string >> 11] + (string & 0x7FF)];
}

byte *GetString(byte *buffr, uint16 string)
{
	uint index = string & 0x7FF;
	uint tab = string >> 11;

	if (string == 0) error("invalid string id 0 in GetString");

	if ( tab == 4 && index >= 0xC0)
		return GetSpecialCityNameString(buffr, index - 0xC0);

	if ( tab == 6 && index == 0xD1)
		return StationGetSpecialString(buffr);

	if ( tab == 14 && index >= 0xE4)
		return GetSpecialPlayerNameString(buffr, index - 0xE4);

	if ( tab == 15)
		return GetName(index, buffr);

	// tab 31 is used for special or dynamic strings
	if ( tab == 31) {
		char *s = NULL;

		switch(index) {
		case 0: s = _station_coverage_text; break;
		case 1: s = _screenshot_name; break;
		case 2: s = _str_300c_string; break;
		}

		return DecodeString(buffr, s);
	}

	if (index >= _langtab_num[tab])
		error("String 0x%X is invalid. Probably because an old version of the .lng file.\n", string);

	return DecodeString(buffr, GetStringPtr(string));
}

void InjectDparam(int amount) 
{
	memmove(_decode_parameters + amount, _decode_parameters, sizeof(_decode_parameters) - amount * sizeof(uint32)); 
}


int32 GetParamInt32()
{
	int32 result = GET_DPARAM32(0);
	memcpy(&_decode_parameters[0], &_decode_parameters[1], sizeof(uint32) * (lengthof(_decode_parameters)-1));
	return result;
}

int64 GetParamInt64()
{
	int64 result = GET_DPARAM32(0) + ((uint64)GET_DPARAM32(1) << 32);
	memcpy(&_decode_parameters[0], &_decode_parameters[2], sizeof(uint32) * (lengthof(_decode_parameters)-2));
	return result;
}


int GetParamInt16()
{
	int result = (int16)GET_DPARAM16(0);
	memcpy(&_decode_parameters[0], &_decode_parameters[1], sizeof(uint32) * (lengthof(_decode_parameters)-1));
	return result;
}

int GetParamInt8()
{
	int result = (int8)GET_DPARAM8(0);
	memcpy(&_decode_parameters[0], &_decode_parameters[1], sizeof(uint32) * (lengthof(_decode_parameters)-1));
	return result;
}

int GetParamUint16()
{
	int result = GET_DPARAM16(0);
	memcpy(&_decode_parameters[0], &_decode_parameters[1], sizeof(uint32) * (lengthof(_decode_parameters)-1));
	return result;
}

static const uint32 _divisor_table[] = {
	1000000000,
	100000000,
	10000000,
	1000000,

	100000,
	10000,
	1000,
	100,
	10,
	1
};

static byte *FormatCommaNumber(byte *buff, int32 number)
{
	uint32 quot,divisor;
	int i;
	uint32 tot;
	uint32 num;

	if (number < 0) {
		*buff++ = '-';
		number = -number;
	}

	num = number;

	tot = 0;
	for(i=0; i!=10; i++) {
		divisor = _divisor_table[i];
		quot = 0;
		if (num >= divisor) {
			quot = num / _divisor_table[i];
			num = num % _divisor_table[i];
		}
		if (tot|=quot || i==9) {
			*buff++ = (byte)('0' + quot);
			if (i==0 || i==3 || i==6) *buff++ = ',';
		}
	}

	*buff = 0;

	return buff;
}

static byte *FormatNoCommaNumber(byte *buff, int32 number)
{
	uint32 quot,divisor;
	int i;
	uint32 tot;
	uint32 num;

	if (number < 0) {
		*buff++ = '-';
		number = -number;
	}

	num = number;

	tot = 0;
	for(i=0; i!=10; i++) {
		divisor = _divisor_table[i];
		quot = 0;
		if (num >= divisor) {
			quot = num / _divisor_table[i];
			num = num % _divisor_table[i];
		}
		if (tot|=quot || i==9) {
			*buff++ = (byte)('0' + quot);
		}
	}

	*buff = 0;

	return buff;
}


static byte *FormatYmdString(byte *buff, uint16 number)
{
	const byte *src;
	YearMonthDay ymd;

	ConvertDayToYMD(&ymd, number);

	for(src = GetStringPtr(ymd.day+STR_01AC_1ST-1); (*buff++=*src++) != 0;) {}
	
	buff[-1] = ' ';
	memcpy(buff, GetStringPtr(STR_0162_JAN + ymd.month), 4);
	buff[3] = ' ';

	return FormatNoCommaNumber(buff+4, ymd.year + 1920);
}


static const char * const _month_names_long[] = {
"January",
"February",
"March",
"April",
"May",
"June",
"July",
"August",
"September",
"October",
"November",
"December",
};

static byte *FormatMonthAndYear(byte *buff, uint16 number)
{
	const char *src;
	YearMonthDay ymd;

	ConvertDayToYMD(&ymd, number);

	for(src = _month_names_long[ymd.month]; (*buff++=*src++) != 0;) {}
	buff[-1] = ' ';

	return FormatNoCommaNumber(buff, ymd.year + 1920);
}

static byte *FormatGenericCurrency(byte *buff, const CurrencySpec *spec, int64 number)
{
	const char *s;
	char c;
	char buf[40], *p;
	int j;

	// multiply by exchange rate
	number *= spec->rate;

	// convert from negative
	if (number < 0) { *buff++ = '-'; number = -number; }
	
	// add pre part
	s = spec->pre;
	while (s != spec->pre + lengthof(spec->pre) && (c=*s++)) *buff++ = c;
	
	// convert to ascii number and add commas
	p = buf;
	j = 4;
	do { 
		if (--j == 0) { *p++ = spec->separator; j = 3; }
		*p++ = '0' + number % 10;
	} while (number /= 10);
	do *buff++ = *--p; while (p != buf);

	// add post part
	s = spec->post;
	while (s != spec->post + lengthof(spec->post) && (c=*s++)) *buff++ = c;

	return buff;
}

static byte *DecodeString(byte *buff, const byte *str)
{
	byte b;

	while((b = *str++) != 0) {
		switch(b) {
		case 0x1: // {SETX}
			*buff++ = b;
			*buff++ = *str++;
			break;
		case 0x1F: // {SETXY}
			*buff++ = b;
			*buff++ = *str++;
			*buff++ = *str++;
			break;
		case 0x7B: // {COMMA32}
			buff = FormatCommaNumber(buff, GetParamInt32());
			break;
		case 0x7C: // {COMMA16}
			buff = FormatCommaNumber(buff, GetParamInt16());
			break;
		case 0x7D: // {COMMA8}
			buff = FormatCommaNumber(buff, GetParamInt8());
			break;
		case 0x7E: // {NUMU16}
			buff = FormatNoCommaNumber(buff, GetParamUint16());
			break;
		case 0x7F: // {CURRENCY}
			buff = FormatGenericCurrency(buff, &_currency_specs[_opt.currency], GetParamInt32());
			break;
		case 0x80: // {STRING}
			buff = GetString(buff, (uint16)GetParamUint16());
			break;
		case 0x81: // {STRINL}
			str += 2;
			buff = GetString(buff, READ_LE_UINT16(str-2));
			break;
		case 0x82: // {DATE_LONG}
			buff = FormatYmdString(buff, GetParamUint16());
			break;
		case 0x83: // {DATE_SHORT}
			buff = FormatMonthAndYear(buff, GetParamUint16());
			break;
		case 0x84: {// {VELOCITY}
			int value = GetParamInt16();
			if (_opt.kilometers) value = value * 1648 >> 10;
			buff = FormatCommaNumber(buff, value);
			if (_opt.kilometers) {
				memcpy(buff, "kmh\xB9", 4);
				buff += 4;
			} else {
				memcpy(buff, "mph", 4);
				buff += 3;
			}
			break;
		}
		case 0x85: // {SKIP16}
			printf("SKIP16 is obsolete\n");
			GetParamInt16();
			break;
		case 0x86: // {SKIP}
			GetParamInt16();
			//assert(0);
			break;
		case 0x87: // {VOLUME}
			buff = FormatCommaNumber(buff, GetParamInt16() * 100);
			memcpy(buff, "liters ", 8);
			buff += 7;
			break;
		case 0x99: { // {CARGO}
			// Layout now is:
			//   8bit   - cargo type
			//   16-bit - cargo count
			int cargo_str = _cargoc.names_long_s[GetParamInt8()];
			// Now check if the cargo count is 1, if it is, increase string by 32.
			if (GET_DPARAM16(0) != 1) cargo_str += 32;
			buff = GetString(buff, cargo_str);
			break;
		case 0x9A: { // {STATION}
			Station *st;
			InjectDparam(1);
			st = DEREF_STATION(GET_DPARAM16(1));
			SET_DPARAM16(0, st->city->citynametype);
			SET_DPARAM32(1, st->city->citynameparts);
			buff = GetString(buff, st->string_id);
			break;
		}
		case 0x9B: { // {CITY}
			City *c;
			c = DEREF_CITY(GET_DPARAM16(0));
			SET_DPARAM32(0, c->citynameparts);
			buff = GetString(buff, c->citynametype);
			break;
		}

		case 0x9C: { // {CURRENCY64}
			buff = FormatGenericCurrency(buff, &_currency_specs[_opt.currency], GetParamInt64());	
			break;
		}

		}
		// case 0x88..0x98: // {COLORS}
		// case 0xE: // {TINYFONT}
		// case 0xF: // {BIGFONT}
		// 0x9E is the highest number that is available.
		default:
			*buff++ = b;
		}
	}
	buff[0] = 0;
	return buff;
}


static byte *StationGetSpecialString(byte *buff)
{
	int x = GetParamInt8();
	if (x & 1) *buff++ = 0xB4;
	if (x & 2) *buff++ = 0xB5;
	if (x & 4) *buff++ = 0xB6;
	if (x & 8) *buff++ = 0xB7;
	if (x & 16) *buff++ = 0xB8;
	*buff = 0;
	return buff;
}

static byte *GetSpecialCityNameString(byte *buff, int ind) {
	uint32 x = GetParamInt32();
	
	switch(ind) {
	case 0: MakeEnglishCityName(buff, x); break;
	case 1: MakeFrenchCityName(buff, x); break;
	case 2: MakeGermanCityName(buff, x); break;
	case 3: MakeAmericanCityName(buff, x); break;
	case 4: MakeSpanishCityName(buff, x); break;
	case 5: MakeSillyCityName(buff, x); break;
	case 6: MakeSwedishCityName(buff, x); break;
	case 7: MakeDutchCityName(buff, x); break;
	default:
		assert(0);
	}
	
	while (*buff != 0) buff++;
	return buff;
}

static const char * const _silly_company_names[] = {
	"Bloggs Brothers",
	"Tiny Transport Ltd.",
	"Express Travel",
	"Comfy-Coach & Co.",
	"Crush & Bump Ltd.",
	"Broken & Late Ltd.",
	"Sam Speedy & Son",
	"Supersonic Travel",
	"Mike's Motors",
	"Lightning International",
	"Pannik & Loozit Ltd.",
	"Inter-City Transport",
	"Getout & Pushit Ltd.",
};

static const char * const _surname_list[] = {
	"Adams",
	"Allan",
	"Baker",
	"Bigwig",
	"Black",
	"Bloggs",
	"Brown",
	"Campbell",
	"Gordon",
	"Hamilton",
	"Hawthorn",
	"Higgins",
	"Green",
	"Gribble",
	"Jones",
	"McAlpine",
	"MacDonald",
	"McIntosh",
	"Muir",
	"Murphy",
	"Nelson",
	"O'Donnell",
	"Parker",
	"Phillips",
	"Pilkington",
	"Quigley",
	"Sharkey",
	"Thomson",
	"Watkins",
	"Grumpy",
	"Dozy",
	"Speedy",
	"Nosey",
	"Dribble",
	"Mushroom",
	"Cabbage",
	"Sniffle",
	"Fishy",
	"Swindle",
	"Sneaky",
	"Nutkins",
};

static const byte _initial_name_letters[] = {
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'P', 'R', 'S', 'T', 'W',
};

static byte *GenAndCoName(byte *buff)
{
	uint32 x = GetParamInt32();
	uint base,num;

	base = 0;
	num = 29;
	if (_opt.landscape == LT_CANDY) {
		base = num;
		num = 12;
	}

	buff = str_cat(buff, _surname_list[base + ((num * (byte)(x >> 16)) >> 8)]);
	buff = str_cat(buff, " & Co.");

	return buff;
}

static byte *GenPlayerName_4(byte *buff)
{
	uint32 x = GetParamInt32();
	uint i, base, num;

	buff[0] = _initial_name_letters[ (sizeof(_initial_name_letters) * (byte)x) >> 8];
	buff[1] = '.';
	buff += 2;

	i = ((sizeof(_initial_name_letters) + 35) * (byte)(x >> 8)) >> 8;
	if (i < sizeof(_initial_name_letters)) {
		buff[0] = _initial_name_letters[i];
		buff[1] = '.';
		buff += 2;		
	}

	base = 0;
	num = 29;
	if (_opt.landscape == LT_CANDY) {
		base = num;
		num = 12;
	}

	buff = str_cat(buff, _surname_list[base + ((num * (byte)(x >> 16)) >> 8)]);

	return buff;
}

static const char * const _song_names[] = {
	"Tycoon DELUXE Theme",
	"Easy Driver",
	"Little Red Diesel",
	"Cruise Control",
	"Don't Walk!",
	"Fell Apart On Me",
	"City Groove",
	"Funk Central",
	"Stoke It",
	"Road Hog",
	"Aliens Ate My Railway",
	"Snarl Up",
	"Stroll On",
	"Can't Get There From Here",
	"Sawyer's Tune",
	"Hold That Train!",
	"Movin' On",
	"Goss Groove",
	"Small Town",
	"Broomer's Oil Rag",
	"Jammit",
	"Hard Drivin'"
};

static byte *GetSpecialPlayerNameString(byte *buff, int ind)
{
	switch(ind) {
	
	// not used
	case 1: {
		int i = GetParamInt32() & 0xFFFF;
		return str_cat(buff, _silly_company_names[i]);
	}

	case 2: // used for Foobar & Co company names
		return GenAndCoName(buff);

	case 3: // President name
		return GenPlayerName_4(buff);

	// song names
	case 4: {
		const char *song = _song_names[GetParamUint16() - 1];
		return str_cat(buff, song);
	}

	case 6: case 7: case 8: case 9: case 10: case 11:
		buff = GetSpecialCityNameString(buff, ind - 6);
		return str_cat(buff, " Transport");
			
	default:
		assert(0);
		return NULL;
	}
}

bool read_language_pack(const char *filename) {
	FILE *in;
	int tot_count, i;
	byte *lang_pack;
	size_t len;
	byte **langpack_offs;
	byte *s;

	in = fopen(filename, "rb");
	if (in == NULL)
		return false;

	fseek(in, 0, SEEK_END);
	len = ftell(in);
	fseek(in, 0, SEEK_SET);
	lang_pack = (byte*)malloc(len + 1); // want a traling NUL
	if (lang_pack == NULL) {
		fclose(in);
		return false;
	}
	lang_pack[len] = 0;
	fread(lang_pack, len, 1, in);
	fclose(in);

#if defined(TTD_BIG_ENDIAN)
	for(i=0; i!=32; i++) {
		((uint16*)lang_pack)[i] = READ_LE_UINT16((uint16*)lang_pack + i);
	}
#endif


	tot_count = 0;
	for(i=0; i!=32; i++) {
		uint num = ((uint16*)lang_pack)[i];
		_langtab_start[i] = tot_count;
		_langtab_num[i] = num;
		tot_count += num;
	}

	// Allocate offsets
	_langpack_offs = langpack_offs = (byte**)malloc(tot_count * sizeof(byte*));

	// Fill offsets
	s = lang_pack + 32 * sizeof(uint16);
	for(i=0; i!=tot_count; i++) {
		len = *s;
		*s++ = 0; // zero terminate the string before.
		if (len >= 0xC0) { len = ((len & 0x3F) << 8) + *s++; }
		langpack_offs[i] = s;
		s += len;
	}
	return true;
}
