#include "../stdafx.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

/* Compiles a list of strings into a compiled string list */

#define lengthof(x) (sizeof(x)/sizeof(x[0]))

typedef void (*ParseCmdProc)(char *buf, int value);

typedef struct CmdStruct {
	const char *cmd;
	ParseCmdProc proc;
	long value;
} CmdStruct;

typedef struct LineName {
	struct LineName *hash_next;
	int value;
	char str[1];
} LineName; 

byte letters[256];
int _cur_line;

#define HASH_SIZE 1023
LineName *_hash_head[HASH_SIZE];
unsigned int hash_str(const char *s) {
	unsigned int hash = 0;
	for(;*s;s++)
		hash = ((hash << 3) | (hash >> 29)) ^ *s;
	return hash % HASH_SIZE;
}
void hash_add(const char *s, int value) {
	unsigned int len = strlen(s);
	LineName *ln = (LineName*)malloc(sizeof(LineName) + len);
	unsigned int hash = hash_str(s);
	ln->hash_next = _hash_head[hash];
	_hash_head[hash] = ln;
	ln->value = value;
	ln->str[len] = 0;
	memcpy(ln->str, s, len);
}

int hash_find(const char *s) {
	LineName *ln = _hash_head[hash_str(s)];
	while (ln != NULL) {
		if (!strcmp(ln->str, s)) {
			return ln->value;
		}
		ln = ln->hash_next;
	}
	return -1;
}

void warning(const char *s, ...) {
	char buf[1024];
	va_list va;
	va_start(va, s);
	vsprintf(buf, s, va);
	va_end(va);
	fprintf(stderr, "%d: WARNING: %s\n", _cur_line, buf);
}

void NORETURN error(const char *s, ...) {
	char buf[1024];
	va_list va;
	va_start(va, s);
	vsprintf(buf, s, va);
	va_end(va);
	fprintf(stderr, "%d: ERROR: %s\n", _cur_line, buf);
	exit(1);
}

char *allstr[65536];

byte _put_buf[2000];
int _put_pos;

void put_byte(byte c) {
	if (_put_pos == lengthof(_put_buf))
		error("Put buffer too small");
	_put_buf[_put_pos++] = c;
}

void emit_buf(int ent) {
	char *s;

	if (ent < 0 || ent >= 0x10000) {
		warning("Invalid string ID %d\n", ent);
		return;
	}

	if (allstr[ent] != 0) {
		warning("Duplicate string ID %d\n", ent);
		return;
	}

	// Allocate the string, and put the uint16 before with the length.
	s = (char*)malloc(sizeof(uint16) + _put_pos);
	*((uint16*)s) = _put_pos;
	memcpy(s + sizeof(uint16), _put_buf, _put_pos);

	allstr[ent] = s;

	_put_pos = 0;
}

void EmitSingleByte(char *buf, int value) {
	if (*buf != 0)
		warning("Ignoring trailing letters in command");
	put_byte((byte)value);
}

void EmitStringInl(char *buf, int value) {
	int id;

	if (*buf>='0' && *buf <= '9') {
		id = strtol(buf, NULL, 0);
		if (id < 0 || id>=0x10000) {
			warning("Invalid inline num %s\n", buf);
			return;
		}
	} else {
		id = hash_find(buf);
		if (id == -1) {
			warning("Invalid inline string '%s'", buf);
			return;
		}
	}

	put_byte(0x81);
	put_byte((byte)(id & 0xFF));
	put_byte((byte)(id >> 8));
}

void EmitSetX(char *buf, int value) {
	char *err;
	int x = strtol(buf, &err, 0);
	if (*err != 0)
		error("SetX param invalid");
	put_byte(1);
	put_byte((byte)x);
}

void EmitSetXY(char *buf, int value) {
	char *err;
	int x = strtol(buf, &err, 0), y;
	if (*err != 0) error("SetXY param invalid");
	y = strtol(err+1, &err, 0);
	if (*err != 0) error("SetXY param invalid");

	put_byte(0x1F);
	put_byte((byte)x);
	put_byte((byte)y);
}

static const CmdStruct _cmd_structs[] = {
	// New line
	{"", EmitSingleByte, 0xD},

	// Font size
	{"TINYFONT", EmitSingleByte, 0xE},
	{"BIGFONT", EmitSingleByte, 0xF},

	// Update position
	{"SETX", EmitSetX, 0x1},
	{"SETXY", EmitSetXY, 0x1F},
	
		// Colors
	{"BLUE", EmitSingleByte, 0x88},
	{"SILVER", EmitSingleByte, 0x89},
	{"GOLD", EmitSingleByte, 0x8A},
	{"RED", EmitSingleByte, 0x8B},
	{"PURPLE", EmitSingleByte, 0x8C},
	{"LTBROWN", EmitSingleByte, 0x8D},
	{"ORANGE", EmitSingleByte, 0x8E},
	{"GREEN", EmitSingleByte, 0x8F},
	{"YELLOW", EmitSingleByte, 0x90},
	{"DKGREEN", EmitSingleByte, 0x91},
	{"CREAM", EmitSingleByte, 0x92},
	{"BROWN", EmitSingleByte, 0x93},
	{"WHITE", EmitSingleByte, 0x94},
	{"LTBLUE", EmitSingleByte, 0x95},
	{"GRAY", EmitSingleByte, 0x96},
	{"DKBLUE", EmitSingleByte, 0x97},
	{"BLACK", EmitSingleByte, 0x98},

	// Numbers
	{"COMMA32", EmitSingleByte, 0x7B},
	{"COMMA16", EmitSingleByte, 0x7C},
	{"COMMA8", EmitSingleByte, 0x7D},
	{"NUMU16", EmitSingleByte, 0x7E},

	{"CURRENCY", EmitSingleByte, 0x7F},
	{"STRING", EmitSingleByte, 0x80},
	{"STRINL", EmitStringInl, 0x81},

	{"DATE_LONG", EmitSingleByte, 0x82},
	{"DATE_SHORT", EmitSingleByte, 0x83},

	{"VELOCITY", EmitSingleByte, 0x84},
	{"SKIP16", EmitSingleByte, 0x85},
	{"SKIP", EmitSingleByte, 0x86},
	{"VOLUME", EmitSingleByte, 0x87},

	{"UPARROW", EmitSingleByte, 0xA0},
	{"POUNDSIGN", EmitSingleByte, 0xA3},
	{"YENSIGN", EmitSingleByte, 0xA5},
	{"COPYRIGHT", EmitSingleByte, 0xA9},
	{"DOWNARROW", EmitSingleByte, 0xAA},
	{"CHECKMARK", EmitSingleByte, 0xAC},
	{"CROSS", EmitSingleByte, 0xAD},
	{"RIGHTARROW", EmitSingleByte, 0xAF},

	{"SMALLUPARROW", EmitSingleByte, 0xBC},
	{"SMALLDOWNARROW", EmitSingleByte, 0xBD},
	{"THREE_FOURTH", EmitSingleByte, 0xBE},
	{"CARGO", EmitSingleByte, 0x99},
	{"STATION", EmitSingleByte, 0x9A},
	{"CITY", EmitSingleByte, 0x9B},
	{"CURRENCY64", EmitSingleByte, 0x9C},
};

const CmdStruct *find_cmd(const char *s) {
	int i;
	const CmdStruct *cs = _cmd_structs;
	for(i=0; i!=lengthof(_cmd_structs); i++,cs++) {
		if (!strcmp(cs->cmd, s))
			return cs;
	}
	return NULL;
}

void handle_string(char *str) {
	char *s,*t;
	const CmdStruct *cs;
	int ent;
		
	// Ignore comments & blank lines
	if (*str == '#' || *str == ' ' || *str == 0)
		return;

	s = strchr(str, ':');
	if (s == NULL) {
		warning("Line has no ':' delimiter");
		return;
	}

	// Trim spaces
	for(t=s;t > str && (t[-1]==' ' || t[-1]=='\t'); t--);
	*t = 0;

	ent = hash_find(str);
	if (ent == -1) {
		warning("String name '%s' is invalid (or already used)", str);
		return;
	}

	s++; // skip :

	for (;*s != 0;s++) {
		if (*s == '{') {
			// Find end of command
			t = ++s;
			for(;;) {
				if (*t == '}') {
					*t = 0;
					break;
				}
				if (*t == ' ') {
					*t = 0;
					t++;
					break;
				}
				if (*t == 0) {
					warning("Newline in the middle of command");
					return;
				}
				t++;
			}

			cs = find_cmd(s);
			if (cs == NULL) {
				warning("No such command '%s'", s);
				return;
			}

			if (*t != 0) {
				// Skip until the end
				for(s=t;;) {
					if (*s == 0) {
						warning("Newline in the middle of command");
						return;
					}
					if (*s == '}') {
						*s = 0;
						break;
					}
					s++;
				}
			} else {
				s = t;
			}


			cs->proc(t, cs->value);
		} else {
			letters[(byte)*s] = 1;
			put_byte(*s);
		}
	}
	emit_buf(ent);
}


#if 0
#define USE_TABLE(x)  { if(index >= lengthof(x)) error("Index %d too big", index); str = x[index]; break; }
const char *get_string(unsigned int string) {
	const char *str;
	unsigned int index = string & 0x7FF;

	switch(string >> 11) {
	case 0:	USE_TABLE(_default_strings)
	case 1: USE_TABLE(_landscape_string_table)
	case 2: USE_TABLE(_track_string_table)
	case 3: USE_TABLE(_road_string_table)
	case 4: 
		USE_TABLE(_city_string_table)
	case 5: USE_TABLE(_tree_string_table)
	case 6: USE_TABLE(_station_string_table)
	case 7: USE_TABLE(_water_string_table)
	case 8: USE_TABLE(_saveload_string_table)
	case 9: USE_TABLE(_industry_string_table)
	case 10: USE_TABLE(_tunnel_string_table)
	case 11: USE_TABLE(_unmovable_string_table)
	case 12: USE_TABLE(_debugger_string_table)
	case 13: USE_TABLE(_newgame_string_table)
	case 14: USE_TABLE(_players_string_table)
	case 16: USE_TABLE(_engines_string_table)
	case 17: USE_TABLE(_trainview_string_table)
	case 18: USE_TABLE(_roadview_string_table)
	case 19: USE_TABLE(_docks_string_table)
	case 20: USE_TABLE(_airport_string_table)
	case 22: USE_TABLE(_disaster_string_table)
	default:
		error("Invalid string grp %d", string >> 11);
	}
	return str;
}


void write_text_line(char *buf, int id, FILE *gen) {
	const char *str = get_string(id);
	int i;
	unsigned int sid;
	
	int chr;

	fprintf(gen, "%-40s:", buf);

	for(;chr=(byte)*str++;) {
		switch(chr) {
		case 0xD: case 0xE: case 0xF:
		case 0x88: case 0x89: case 0x8A: case 0x8B:
		case 0x8C: case 0x8D: case 0x8E: case 0x8F:
		case 0x90: case 0x91: case 0x92: case 0x93:
		case 0x94: case 0x95: case 0x96: case 0x97:
		case 0x98:
		case 0x7B: case 0x7C: case 0x7D: case 0x7E:
		case 0x7F: case 0x80: case 0x82: 
		case 0x83: case 0x84: case 0x85: case 0x87:
			{
			const CmdStruct *cs = _cmd_structs;
			for(i=0; ; i++,cs++) {
				if (i==lengthof(_cmd_structs)) {
					warning("Found no matching entry for 0x%X\n", chr);
					break;
				}
				if (cs->proc == EmitSingleByte && cs->value == chr) {
					fprintf(gen,"{%s}", cs->cmd);
					break;
				}
			}
			}
			break;

		case 0x81:
			sid = (byte)str[0] | ((byte)str[1]<<8);
			str+=2;
			fprintf(gen,"{STRINL 0x%X}", sid);
			break;

		case 1:
			fprintf(gen, "{SETX %d}", (byte)str[0]);
			str+=1;
			break;

		case 0x1F:
			fprintf(gen, "{SETXY %d %d}", (byte)str[0], (byte)str[1]);
			str+=2;
			break;

		default:
			if (chr < 32)
				error("Character %d < 32 in %s", chr, buf);
			fputc(chr, gen);
		}
	}
	fputs("\n", gen);
}
#endif

void parse_strings_h(char *s, FILE *gen) {
	FILE *f = fopen(s, "r");
	char buf[2048];
	int i, cur_id;

	if (f == NULL) {
		error("Cannot open strings.h");
	}

	_cur_line = 1;

	cur_id = -1;

	while (fgets(buf, sizeof(buf), f) != NULL) {
		i = strlen(buf);
		while (i>0 && (buf[i-1]=='\r' || buf[i-1]=='\n' || buf[i-1] == ' ')) i--;
		buf[i] = 0;

		if (i == 0 || 
				memcmp(buf,"enum",4) == 0 || 
				memcmp(buf,"};", 2) == 0)
					continue;

		if (buf[i-1] != ',') {
			warning("Line lacks a trailing ','");
			continue;
		}

		cur_id++;

		s = strchr(buf, '=');
		if (s != NULL) {
			while (*++s == ' ');
			cur_id = strtol(s, NULL, 0);
		}

		for(s=buf; *s != ',' && *s != ' '; s++);
		*s = 0;

		if ((cur_id & 0xf800) != 0xF800)
			hash_add(buf, cur_id);

#if 0
		if (gen != NULL)
			write_text_line(buf, cur_id, gen);
#endif
		_cur_line++;
	}

	fclose(f);
}

void parse_stdin(FILE *in) {
	char buf[2048];
	int i;
	
	_cur_line = 1;
	while (fgets(buf, sizeof(buf),in) != NULL) {
		i = strlen(buf);
		while (i>0 && (buf[i-1]=='\r' || buf[i-1]=='\n' || buf[i-1] == ' ')) i--;
		buf[i] = 0;

		handle_string(buf);
		_cur_line++;
	}
}

int count_inuse(int grp) {
	int i,j;

	for(i=0; i != 0x800; i++) {
		if (allstr[(grp<<11)+i] == NULL) {
			for(j=i; j != 0x800; j++) {
				if (allstr[(grp<<11)+j] != NULL) {
					warning("Group %d has hole in it at %d", i, j);
					break;
				}
			}
			break;
		}
	}
	return i;
}

void check_all_used() {
	int i;
	LineName *ln;
	int num_warn = 10;

	for (i=0; i!=HASH_SIZE; i++) {
		for(ln = _hash_head[i]; ln!=NULL; ln = ln->hash_next) {
			if (allstr[ln->value] == 0) {
				if (++num_warn < 50) {
					warning("String %s has no definition. Using NULL value", ln->str);
				}
				_put_pos = 0;
				emit_buf(ln->value);
			}
		}
	}
}

void write_length(FILE *f, uint length)
{
	if (length < 0xC0) {
		fputc(length, f);
	} else if (length < 0x4000) {
		fputc((length >> 8) | 0xC0, f);
		fputc(length & 0xFF, f);
	} else {
		error("string too long");
	}
}

void gen_output(FILE *f) {
	uint16 in_use[32];
	uint16 in_use_file[32];
	int i,j;
	int tot_str = 0;

	check_all_used();

	for(i=0; i!=32; i++) {
		int n = count_inuse(i);
		in_use[i] = n;
		in_use_file[i] = TO_LE16(n);
		tot_str += n;
	}

	fwrite(in_use_file, 32*sizeof(uint16), 1, f);

	for(i=0; i!=32; i++) {
		for(j=0; j!=in_use[i]; j++) {
			char *s = allstr[(i<<11)+j];
			if (s == NULL) error("Internal error, s==NULL");
			
			write_length(f, *(uint16*)s);
			fwrite(s + sizeof(uint16), *(uint16*)s , 1, f);
			tot_str--;
		}
	}

	fputc(0, f); // write trailing nul character.

	if (tot_str != 0) {
		error("Internal error, tot_str != 0");
	}
}


int main(int argc, char* argv[])
{
	FILE *f;

	parse_strings_h("table/strings.h", NULL);
	
	f = fopen(argv[2], "r");
	if (f == NULL)
		error("cannot open input file '%s'", argv[2]);
	parse_stdin(f);
	fclose(f);

	f = fopen(argv[1], "wb");
	if (f == NULL)
		error("cannot open output file '%s'", argv[1]);
	gen_output(f);
	fclose(f);

	return 0;
}

