/* Copyright (c) 1995 by Internet Software Consortium
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* eventlib.c - implement clue for the eventlib
 * vix 09sep95 [initial]
 */

#if !defined(LINT) && !defined(CODECENTER)
static const char rcsid[] = "$Id: eventlib.c,v 1.2 1995/09/17 23:12:25 vixie Exp $";
#endif

#include <assert.h>

#include "eventlib.h"
#include "eventlib_p.h"

static evEvent_p *GetSignal(evContext_p *ctx);
static const struct timeval NoTime = {0, 0};
static const struct timeval ShortTime = {1, 500000};

int
evOpen(evContext *opaqueCtx) {
	evContext_p *ctx;

	OKNEW(ctx);

	/* Timers. */
	ctx->timers = NULL;

	/* Files. */
	ctx->files = NULL;
	FD_ZERO(&ctx->rdNext);
	FD_ZERO(&ctx->wrNext);
	FD_ZERO(&ctx->exNext);
	FD_ZERO(&ctx->nonblockBefore);
	ctx->fdMax = -1;
	ctx->fdNext = NULL;
	ctx->fdCount = 0;	/* Invalidate {rd,wr,ex}Last. */

	/* Signals. */
	ctx->signals = NULL;
	sigemptyset(&ctx->blockedBefore);

	opaqueCtx->opaque = ctx;
	return (0);
}

int
evClose(evContext opaqueCtx) {
	evContext_p *ctx = opaqueCtx.opaque;
	int revs = 424242;	/* Doug Adams. */

	/* Timers. */
	while (revs-- > 0 && ctx->timers != NULL) {
		evTimerID timer;

		timer.opaque = ctx->timers;
		(void) evClearTimer(opaqueCtx, timer);
	}

	/* Files. */
	while (revs-- > 0 && ctx->files != NULL) {
		evFileID file;

		file.opaque = ctx->files;
		(void) evDeselectFD(opaqueCtx, file);
	}

	/* Signals. */
	while (revs-- > 0 && ctx->signals != NULL) {
		evSignalID sig;

		sig.opaque = ctx->signals;
		(void) evClearSignal(opaqueCtx, sig);
	}

	free(ctx);
	return(0);
}

int
evGetNext(evContext opaqueCtx, evEvent *opaqueEv, int options) {
	evContext_p *ctx = opaqueCtx.opaque;
	struct timeval now, nextTime;
	evTimer *nextTimer;
	evEvent_p *new;
	int x, timerPast;

	/* Ensure that exactly one of EV_POLL or EV_WAIT was specified. */
	x = ((options & EV_POLL) != 0) + ((options & EV_WAIT) != 0);
	if (x != 1)
		ERR(EINVAL);

	/* Give signals top priority. */
	if ((new = GetSignal(ctx)) != NULL) {
		opaqueEv->opaque = new;
		return (0);
	}

	/* Get the time of day.  We'll do this again after select() blocks. */
	OK(gettimeofday(&now, NULL));

	/* Get the status and content of the next timer. */
	if ((nextTimer = ctx->timers) != NULL) {
		nextTime = nextTimer->due;
		timerPast = (evCmpTimeV(nextTime, now) <= 0);
	}

 again:
	DPRINTF(9, ("evGetNext: again: fdCount %d\n", ctx->fdCount));
	if (!ctx->fdCount) {
		enum { JustPoll, ShortPoll, Block, Timer } m;
		struct timeval t, *tp;

		/* Are there any events at all? */
		if ((options & EV_WAIT) && !nextTimer && ctx->fdMax == -1 &&
		    !ctx->signals)
			ERR(ENOENT);

		/* Figure out what select()'s timeout parameter should be. */
		if (options & EV_POLL) {
			m = JustPoll;
			t = NoTime;
			tp = &t;
		} else if (! nextTimer) {
			if (ctx->signals) {
				m = ShortPoll;
				t = ShortTime;
				tp = &t;
			} else {
				m = Block;
				/* ``t'' unused. */
				tp = NULL;
			}
		} else if (timerPast) {
			m = JustPoll;
			t = NoTime;
			tp = &t;
		} else {
			m = Timer;
			/* ``t'' filled in later. */
			tp = &t;
		}

		do {
			/* We do these here due to possible EINTR. */
			ctx->rdLast = ctx->rdNext;
			ctx->wrLast = ctx->wrNext;
			ctx->exLast = ctx->exNext;

			/* We do this here due to possible EINTR. */
			if (m == Timer) {
				assert(tp == &t);
				t = evSubTimeV(nextTime, now);
				/* Do we need to ensure periodic sig polls? */
				if (ctx->signals != NULL &&
				    evCmpTimeV(t, ShortTime) > 0)
					t = ShortTime;
			}

			DPRINTF(4, ("select(%d, %#x, %#x, %#x, %d.%06d)\n",
				    ctx->fdMax+1,
				    &ctx->rdLast, &ctx->wrLast, &ctx->exLast,
				    tp ? tp->tv_sec : -1,
				    tp ? tp->tv_usec : -1));

			x = select(ctx->fdMax+1,
				   &ctx->rdLast, &ctx->wrLast, &ctx->exLast,
				   tp);

			/* Did any signals come in during select()? */
			if ((new = GetSignal(ctx)) != NULL) {
				opaqueEv->opaque = new;
				return (0);
			}

			/* Anything but a poll can change the time. */
			if (m != JustPoll)
				OK(gettimeofday(&now, NULL));

			/* Select() likes to finish about 10ms early. */
			if (x == 0 && m == Timer &&
			    evCmpTimeV(now, nextTime) < 0) {
				/* Simulate EINTR, go back, select() again. */
				x = -1;
				errno = EINTR;
			}

			/* Are we polling for a signal? */
			if (x == 0 && m == ShortPoll && ctx->signals) {
				/* Simulate EINTR, go back, select() again. */
				x = -1;
				errno = EINTR;
			}
		} while (x < 0 && errno == EINTR);
		if (x < 0)
			ERR(errno);
		if (x == 0 && !timerPast && (options & EV_POLL))
			ERR(EWOULDBLOCK);
		ctx->fdCount = x;
	}
	assert(nextTimer || ctx->fdCount);

	/* Timers go first since we'd like them to be accurate. */
	if (nextTimer && !timerPast) {
		/* Has anything happened since we blocked? */
		timerPast = (evCmpTimeV(nextTime, now) <= 0);
	}
	if (nextTimer && timerPast) {
		OKNEW(new);
		new->type = Timer;
		new->u.timer.this = nextTimer;
		opaqueEv->opaque = new;
		return (0);
	}

	/* No timers, so there must be a ready file descriptor. */
	while (ctx->fdCount) {
		register evFile *fid;
		register int fd, eventmask;

		if (!ctx->fdNext)
			ctx->fdNext = ctx->files;
		fid = ctx->fdNext;
		ctx->fdNext = fid->next;

		fd = fid->fd;
		eventmask = 0;
		if (FD_ISSET(fd, &ctx->rdLast))
			eventmask |= EV_READ;
		if (FD_ISSET(fd, &ctx->wrLast))
			eventmask |= EV_WRITE;
		if (FD_ISSET(fd, &ctx->exLast))
			eventmask |= EV_EXCEPT;
		if (eventmask != 0) {
			ctx->fdCount--;
			OKNEW(new);
			new->type = File;
			new->u.file.this = fid;
			new->u.file.eventmask = eventmask;
			opaqueEv->opaque = new;
			return (0);
		}
	}

	/* We get here if the caller deselect()'s an FD. Gag me with a goto. */
	goto again;
}

static evEvent_p *
GetSignal(evContext_p *ctx) {
	evEvent_p *new;
	evSignal *cur;
	sigset_t set;

	if (sigpending(&set) < 0)
		return (NULL);
	for (cur = ctx->signals; cur; cur = cur->next)
		if (sigismember(&set, cur->sig))
			break;
	if (!cur)
		return (NULL);

	/*
	 * Briefly unblock and then reblock the signal.  According to
	 * IEEE Std1003.1-1988 ("POSIX") 3.3.5.1, at least one signal
	 * will be delivered before a return from an unblocking call
	 * to setprocmask().  Since we're only unblocking one signal,
	 * and it's one that sigpending() says is pending (see above),
	 * this ought to clear the signal (and the sigpending().)  We
	 * immediately turn around and reblock this signal afterward.
	 */
	sigemptyset(&set);
	sigaddset(&set, cur->sig);
	(void) sigprocmask(SIG_UNBLOCK, &set, NULL);
	(void) sigpending(&set);
	assert(sigismember(&set, cur->sig) <= 0);
	sigemptyset(&set);
	sigaddset(&set, cur->sig);
	(void) sigprocmask(SIG_BLOCK, &set, NULL);

	NEW(new);
	if (!new) {
		/* Note: we're silently losing a signal in this case. */
		errno = ENOMEM;
		return (NULL);
	}
	new->type = Signal;
	new->u.signal.this = cur;
	return (new);
}

int
evDispatch(evContext opaqueCtx, evEvent opaqueEv) {
	evContext_p *ctx = opaqueCtx.opaque;
	evEvent_p *ev = opaqueEv.opaque;

	switch (ev->type) {
	    case Timer: {
		evTimer *this = ev->u.timer.this;

		(this->func)(opaqueCtx, this->uap, this->due, this->inter);
		if (this->inter.tv_sec == 0 && this->inter.tv_usec == 0) {
			evTimerID opaque;

			opaque.opaque = this;			
			(void) evClearTimer(opaqueCtx, opaque);
		} else {
			this->due = evAddTimeV(this->due, this->inter);
			/* XXX a signal that called evSetTimer() would hurt. */
			evRelinkTimers(ctx, this, this);
		}
		break;
	    }
	    case File: {
		evFile *this = ev->u.file.this;
		int eventmask = ev->u.file.eventmask;

		(this->func)(opaqueCtx, this->uap, this->fd, eventmask);
		break;
	    }
	    case Signal: {
		evSignal *this = ev->u.signal.this;

		(this->func)(opaqueCtx, this->uap, this->sig);
		break;
	    }
	    default: {
		abort();
	    }
	}
	return (0);
}

int
evMainLoop(evContext opaqueCtx) {
	evContext_p *ctx = opaqueCtx.opaque;
	evEvent event;
	int x;

	while ((x = evGetNext(opaqueCtx, &event, EV_WAIT)) == 0)
		if ((x = evDispatch(opaqueCtx, event)) < 0)
			break;
	return (x);
}

