/*
 * DMW96 rfkill power control via GPIO
 *
 * Copyright (C) 2011 DSP Group
 * 
 *  Initial code: Pavan Savoy <pavan.savoy@gmail.com> (wl127x_power.c)
 *  modifed by: Ziv Haziz (ziv.haziz@dspg.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <linux/dmw96-rfkill.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/rfkill.h>
#include <linux/slab.h>

struct dmw96_rfkill {
	struct dmw96_rfkill_platform_data *pdata;
	struct rfkill *bt_rfkill;
};

static int dmw96_bt_set_block(void *data, bool blocked)
{
	struct dmw96_rfkill_platform_data *pdata = data;

	if (gpio_is_valid(pdata->btshutdown_gpio))
		gpio_set_value(pdata->btshutdown_gpio, blocked ? 0:1);

	return 0;
}

static const struct rfkill_ops dmw96_bt_rfkill_ops = {
	.set_block = dmw96_bt_set_block,
};

static int dmw96_rfkill_probe(struct platform_device *pdev)
{
	int err = 0;
	struct dmw96_rfkill_platform_data *pdata = pdev->dev.platform_data;
	struct dmw96_rfkill *priv;

	if (!pdata) {
		dev_err(&pdev->dev, "missing platform data!\n");
		return -EINVAL;
	}

	priv = kzalloc(sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	platform_set_drvdata(pdev, priv);
	priv->pdata = pdata;

	if (gpio_is_valid(pdata->btshutdown_gpio)) {
		err = gpio_request(pdata->btshutdown_gpio, "dmw96_btshutdown_gpio");
		if (unlikely(err)) {
			dev_err(&pdev->dev, "cannot get GPIO!\n");
			goto err_free;
		}

		err = gpio_direction_output(pdata->btshutdown_gpio, 0);
		if (unlikely(err)) {
			dev_err(&pdev->dev, "cannot set GPIO!\n");
			goto err_free_gpio;
		}
	}

	priv->bt_rfkill = rfkill_alloc("dmw96-bluetooth", &pdev->dev,
					RFKILL_TYPE_BLUETOOTH,
					&dmw96_bt_rfkill_ops,
					pdata);

	if (unlikely(!priv->bt_rfkill)) {
		err = -ENOMEM;
		goto err_free_gpio;
	}

	/* bluetooth starts in rf off -> blocked == true */
	rfkill_init_sw_state(priv->bt_rfkill, true);

	/* api for in kernel functions to stop and start the rf */
	rfkill_set_hw_state(priv->bt_rfkill, false);

	err = rfkill_register(priv->bt_rfkill);
	if (err)
		goto err_free_rfkill;

	if (gpio_is_valid(pdata->btshutdown_gpio))
		dev_info(&pdev->dev, "using GPIO %u\n", pdata->btshutdown_gpio);
	else
		dev_info(&pdev->dev, "using no GPIO\n");

	return 0;

err_free_rfkill:
	rfkill_destroy(priv->bt_rfkill);
err_free_gpio:
	if (gpio_is_valid(pdata->btshutdown_gpio))
		gpio_free(pdata->btshutdown_gpio);
err_free:
	kfree(priv);
	return err;
}

static int dmw96_rfkill_remove(struct platform_device *pdev)
{
	struct dmw96_rfkill *priv = platform_get_drvdata(pdev);

	rfkill_unregister(priv->bt_rfkill);
	rfkill_destroy(priv->bt_rfkill);
	if (gpio_is_valid(priv->pdata->btshutdown_gpio))
		gpio_free(priv->pdata->btshutdown_gpio);
	kfree(priv);

	return 0;
}

static struct platform_driver dmw96_rfkill_platform_driver = {
	.probe = dmw96_rfkill_probe,
	.remove = dmw96_rfkill_remove,
	.driver = {
		.name = "dmw96_rfkill",
		.owner = THIS_MODULE,
	},
};

static int __init dmw96_rfkill_init(void)
{
	return platform_driver_register(&dmw96_rfkill_platform_driver);
}

static void __exit dmw96_rfkill_exit(void)
{
	platform_driver_unregister(&dmw96_rfkill_platform_driver);
}

module_init(dmw96_rfkill_init);
module_exit(dmw96_rfkill_exit);

MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");
