/*
 * dmw96ciu_w.h
 *
 * Copyright (C) 2010 DSPG
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef DMW96CAM_H
#define DMW96CAM_H

#include <media/v4l2-int-device.h>
#include <media/videobuf-core.h>

#define CAM_NAME			"dmw96ciu"


#define DMW96_CIU_AF     	(1 << 4)
#define DMW96_CIU_HIST   	(1 << 5)
#define DMW96CAM_XCLK_NONE	-1
#define DMW96CAM_XCLK_A	0
#define DMW96CAM_XCLK_B	1

#define DMW96CAM_SLAVE_SENSOR	0
#define DMW96CAM_SLAVE_LENS		1
#define DMW96CAM_SLAVE_FLASH		2 /* This is the last slave! */

/* mask for dmw96cam_slave_power_set */
#define DMW96CAM_SLAVE_POWER_SENSOR	(1 << DMW96CAM_SLAVE_SENSOR)
#define DMW96CAM_SLAVE_POWER_LENS	(1 << DMW96CAM_SLAVE_LENS)
#define DMW96CAM_SLAVE_POWER_SENSOR_LENS \
	(DMW96CAM_SLAVE_POWER_SENSOR | DMW96CAM_SLAVE_POWER_LENS)
#define DMW96CAM_SLAVE_POWER_FLASH	(1 << DMW96CAM_SLAVE_FLASH)
#define DMW96CAM_SLAVE_POWER_ALL	-1

#define DMW96CAM_VIDEODEVS		4

#define DMW96CAM_POWEROFF_DELAY (1 * HZ)

struct dmw96cam_device;
struct dmw96cam_videodev;


struct dmw96cam_sensor_config {
	int xclk;
};

struct dmw96cam_lens_config {
};

struct dmw96cam_flash_config {
};

/**
 * struct dmw96cam_hw_config - struct for vidioc_int_g_priv ioctl
 * @xclk: DMW96CAM_XCLK_A or DMW96CAM_XCLK_B
 * @s_pix_sparm: Access function to set pix and sparm.
 * Pix will override sparm
 */
struct dmw96cam_hw_config {
	int dev_index; /* Index in dmw96cam_sensors */
	int dev_minor; /* Video device minor number */
	int dev_type; /* DMW96CAM_SLAVE_* */
	union {
		struct dmw96cam_sensor_config sensor;
		struct dmw96cam_lens_config lens;
		struct dmw96cam_flash_config flash;
	} u;
};

/**
 * struct dmw96cam_videodev - per /dev/video* structure
 * @mutex: serialises access to this structure
 * @cam: pointer to cam hw structure
 * @master: we are v4l2_int_device master
 * @sensor: sensor device
 * @lens: lens device
 * @flash: flash device
 * @slaves: how many slaves we have at the moment
 * @vfd: our video device
 * @index: index of this structure in cam->vdevs
 * @users: how many users we have
 * @power_state: Current power state
 * @power_state_wish: New power state when poweroff_timer expires
 * @power_state_mask: Bitmask of devices to set the new power state
 * @poweroff_timer: Timer for dispatching poweroff_work
 * @poweroff_work: Work for slave power state change
 * @sensor_config: CIU-speicific sensor configuration
 * @lens_config: CIU-speicific lens configuration
 * @flash_config: CIU-speicific flash configuration
 * @timeperframe: Desired timeperframe
 * @pix: Desired pix
 * @streaming: streaming file handle, if streaming is enabled
 */
struct dmw96cam_videodev {
	struct mutex mutex; /* serialises access to this structure */

	struct dmw96cam_device *cam;
	struct v4l2_int_device master;

#define vdev_sensor slave[DMW96CAM_SLAVE_SENSOR]
#define vdev_lens slave[DMW96CAM_SLAVE_LENS]
#define vdev_flash slave[DMW96CAM_SLAVE_FLASH]
	struct v4l2_int_device *slave[DMW96CAM_SLAVE_FLASH + 1];

	/* number of slaves attached */
	int slaves;

	/*** video device parameters ***/
	struct video_device *vfd;

	/*** general driver state information ***/
	int index;
	atomic_t users;
	enum v4l2_power power_state[DMW96CAM_SLAVE_FLASH + 1];
#ifdef DMW96CAM_POWEROFF_DELAY
	enum v4l2_power power_state_wish;
	int power_state_mask;
	struct timer_list poweroff_timer;
	struct work_struct poweroff_work;
#endif /* DMW96CAM_POWEROFF_DELAY */

#define vdev_sensor_config slave_config[DMW96CAM_SLAVE_SENSOR].u.sensor
#define vdev_lens_config slave_config[DMW96CAM_SLAVE_LENS].u.lens
#define vdev_flash_config slave_config[DMW96CAM_SLAVE_FLASH].u.flash
	struct dmw96cam_hw_config slave_config[DMW96CAM_SLAVE_FLASH + 1];

	/*** capture data ***/
	struct file *streaming;

	struct v4l2_pix_format pix;
	struct v4l2_fract timeperframe;
};

/**
 * struct dmw96cam_device - per-device data structure
 * @mutex: mutex serialises access to this structure
 * @dev: device structure
 * @vdevs: /dev/video specific structures
 */
struct dmw96cam_device {
	struct mutex mutex; /* serialises access to this structure */

	/*** interfaces and device ***/
	struct dmw96cam_videodev vdevs[DMW96CAM_VIDEODEVS];
	struct device *dev;

};

#endif /* ifndef DMW96CAM_H */
