/*
 * TDM+DMA test case with WM8976 for EVB board.
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/delay.h>
#include <linux/dmw96dma.h>
#include <linux/err.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/clk.h>

#include <mach/hardware.h>
#include <mach/irqs.h>
#include <mach/reset.h>

/**
 * Choose your configuration...
 */
#define SAMPLE_FREQ 44100
#define BUFFER_SIZE (1024*8)
#define TDM_FORMAT_I2S
#define WOLFSON_IS_MASTER
#define OUTPUT_ENABLE
#define OUTPUT_USE_DMA
//#define INPUT_ENABLE
//#define INPUT_USE_DMA

#define DP_CLK_FREQ	12000000

#define TDM_REG_BASE	DMW_TDM3_BASE
#define TDM_REG_SIZE	(0x4000)
#define TDM_IID		DMW_IID_TDM3
#define TDM_DMA_RX	12
#define TDM_DMA_TX	13
#define TDM_RESET	RESET_TDM3
#define TDM_CLOCK	"dmw-tdm.2"

/* TDM registers: */
#define TDM_ENABLE      0x0000
#define TDM_CONFIG_1    0x0004
#define TDM_CONFIG_1_AUTO_SYNC       (1 << 13)
#define TDM_CONFIG_1_CLK_MSTR_SLV    (1 << 12)
#define TDM_CONFIG_1_FSYNC_MSTR_SLV  (1 << 11)
#define TDM_CONFIG_1_SYNC_EDGE       (1 << 10)
#define TDM_CONFIG_1_RX_EDGE         (1 <<  9)
#define TDM_CONFIG_1_TX_EDGE         (1 <<  8)
#define TDM_CONFIG_1_TIME_SLOTS      0x1F
#define TDM_CONFIG_2    0x0008
#define TDM_CONFIG_2_SHIFT      (1 << 9)
#define TDM_CONFIG_2_TRUNCATE   (1 << 8)
#define TDM_CONFIG_2_BANK_SIZE  0x07
#define TDM_FSYNC       0x0010
#define TDM_FSYNC_DUR   0x0014
#define TDM_FIFO_WM     0x0018
#define TDM_INT_EN      0x0020
#define TDM_INT_CLR     0x0028
#define TDM_INT_STATUS  0x002C
#define TDM_STATUS      0x0030
#define TDM_TX_FIFO     0x0038
#define TDM_RX_FIFO     0x003C

/* Less then 32: Give Earlier watermark interrupt - less chance for Underrun/Overrun */
#define TDM_TX_FIFO_LEVEL  31
#define TDM_RX_FIFO_LEVEL  31


struct tdm {
	void *reg_base;

	struct clk *dp_clk;
	struct clk *tdm_clk;
	int running;

	void *buffer;
	void *buffer_end;
	u32 *rx_ptr;
	u32 *tx_ptr;

	struct dmw96dma_list *dma_list_out;
	struct dmw96dma_list *dma_list_in;
};

#define WM8976_NUM_REGS		58
#define WM8976_INVALID_REG_ADDR	0xFFFF //invalid register address

#define WM8976_RESET			 0
#define WM8976_POWER_MGMT_1		 1
#define WM8976_POWER_MGMT_2		 2
#define WM8976_POWER_MGMT_3		 3
#define WM8976_TDM_FORMAT		 4
#define WM8976_ADC_DAC_COMPAND		 5
#define WM8976_CLK_CTRL			 6
#define WM8976_APPROX_SR		 7
#define WM8976_IN_ADC_VOL		15
#define WM8976_OUT_LDAC_VOL		11
#define WM8976_OUT_RDAC_VOL		12
#define WM8976_PLL_N			36
#define WM8976_PLL_K_1			37
#define WM8976_PLL_K_2			38
#define WM8976_PLL_K_3			39
#define WM8976_BEEP_MIXER		43
#define WM8976_IN_PGA_CTRL		44
#define WM8976_IN_PGA_VOL		45
#define WM8976_IN_ADC_BOOST_CTRL	47
#define WM8976_OUT_ANLG_MIXER		49
#define WM8976_LOUT_ANLG_MIXER		50
#define WM8976_ROUT_ANLG_MIXER		51
#define WM8976_LOUT1_VOL		52
#define WM8976_ROUT1_VOL		53
#define WM8976_LOUT2_VOL		54
#define WM8976_ROUT2_VOL		55
#define WM8976_OUT3_ANLG_MIXER		56
#define WM8976_OUT4_ANLG_MIXER		57

static s16 sine[] = {
	     0,
	  6269,
	 11584,
	 15136,
	 16383,
	 15137,
	 11586,
	  6271,
	     1,
	 -6268,
	-11583,
	-15136,
	-16383,
	-15137,
	-11587,
	 -6272
};

static u16 wm8976_regs[WM8976_NUM_REGS] = {
	0x0000, 0x00FF, 0x0005, 0x000F,	/*  0 -  3 */
#ifdef TDM_FORMAT_I2S
	0x0010,				/*  4 */	// Standard I2S format
#else
	0x0008,				/*  4 */	// Standard Left-Justified format
#endif
		0x0000, 0x014C, 0x0000,	/*  5 -  7 */
	0x0000, 0x0000, 0x0000, 0x01FF,	/*  8 - 11 */
	0x01FF, 0x0000, 0x0100, 0x01FF,	/* 12 - 15 */
	0xFFFF, 0xFFFF, 0x002C, 0x002C,	/* 16 - 19 */ // 16-Reserved, 17-NonExist
	0x002C, 0x002C, 0x002C, 0xFFFF,	/* 20 - 23 */ // 23-NonExist
	0x0132, 0x0000, 0xFFFF, 0x0000,	/* 24 - 27 */ // 26-NonExist
	0x0000, 0x0000, 0x0000, 0xFFFF,	/* 28 - 31 */ // 31-NonExist
	0x0038, 0x000B, 0x0032, 0x0000,	/* 32 - 35 */
	0x0007, 0x0021, 0x0161, 0x0026,	/* 36 - 39 */
	0xFFFF, 0x0000, 0xFFFF, 0x0010,	/* 40 - 43 */ // 40-Reserved, 42-NonExist
	0x0003, 0x019F, 0xFFFF, 0x0100,	/* 44 - 47 */ // 46-Reserved
	0xFFFF, 0x0002, 0x0001, 0x0001,	/* 48 - 51 */ // 48-Reserved
	0x0179, 0x0179, 0x0179, 0x0179,	/* 52 - 55 */
	0x0041, 0x0041			/* 56 - 57 */
};

struct snd_wm8976_partial_palette {
	unsigned int	addr;
	unsigned int	mask; // '1' - bit to change; '0' bit to keep !
	unsigned int	value;
};

static struct snd_wm8976_partial_palette broadtile_music[] = {

	// Outputs:
	// -------

	/* Enable LOUT1, ROUT1 (power-save) */
	{WM8976_POWER_MGMT_2, 0x0180, 0x0180},	// Reg 2: enable LOUT1EN, ROUT1EN

	/* Set LDAC -> L MIXER -> LOUT1 */
	{WM8976_LOUT_ANLG_MIXER, 0x0023 ,0x0001},	// Reg 50: enable DACL2LMIX , disable BYPL2LMIX, AUXL2LMIX

	/* Set RDAC -> R MIXER -> ROUT1 */
	{WM8976_ROUT_ANLG_MIXER, 0x0021 ,0x0001},	// Reg 51: enable DACR2RMIX , disable AUXR2RMIX

	/* Disable cross inputs to mixers */
	{WM8976_OUT_ANLG_MIXER, 0x0060, 0x0000},	// Reg 49: disable DACL2RMIX, DACR2LMIX

	/* Unmute LOUT1, ROUT1 */
	{WM8976_LOUT1_VOL, 0x0040, 0x0000},		// Reg 52: unmute LOUT1
	{WM8976_ROUT1_VOL, 0x0040, 0x0000},		// Reg 53: unmute ROUT1


	// Input:
	// -----
	/* A.M. Note:
	 * In the BTP board of the Broadtile, I have changed the microphone circutry to get a better
	 * mic siganal:
	 * - mic Bias resistors replaced, from 1K/1K to 0.39K/2.2K Ohm.
	 * - the 10uF power-stabilizer capacitor removed; it shunt the signal.
	 * As these changes are done, no extra 20db boost is needed, and gain of ~ 12db is enough.
	 */
	/* Config to Single-Ended microphone: PGA Amp inputs: Connect LIN, Disconnect LIP, Disconnect L2 */
	//{WM8976_IN_PGA_CTRL, 0x0007, 0x0002}, // Reg 44: enable LIN2INPPGA, disable LIP2INPPGA, L2_2INPPGA

	/* Config to Differential microphone: PGA Amp inputs: Connect LIP & LIN, Disconnect L2 */
	{WM8976_IN_PGA_CTRL, 0x0007, 0x0003}, // Reg 44: enable LIP2INPPGA, LIN2INPPGA, disable L2_2INPPGA

	/* Enable mic Bias */
	{WM8976_POWER_MGMT_1, 0x0010, 0x0010}, // Reg 1: enable MICBEN

	/* Boost input stage by +20db */
	{WM8976_IN_ADC_BOOST_CTRL, 0x0100, 0x0100},	// Reg 47: enable PGABOOSTL
	{WM8976_POWER_MGMT_2, 0x0014, 0x0014},		// Reg  2: enable INPPGAENL, BOOSTENL

	/* Unmute mic and set mic gain: ~ +12db */
	{WM8976_IN_PGA_VOL, 0x01FF, 0x0120},	// Reg 45: unmute INPPGAMUTEL + Volume

	/* Terminator */
	{WM8976_INVALID_REG_ADDR, 0, 0}
};

#define F2_8000_OR_12000_FAMILY	98304000ull
#define F2_11025_FAMILY		90316800ull
#define FLOAT_MULTIPLYER	10000

unsigned long long dp_clk_freq;

static void wm8976_set_pll(void)
{
	unsigned long long	r;
	unsigned long long	k;
	unsigned long long	r_rest;
	unsigned int		plln = 0;
	unsigned int		r_int;

#if SAMPLE_FREQ == 8000
	r = F2_8000_OR_12000_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x1EC;
	wm8976_regs[WM8976_APPROX_SR] = 0x00A;
#elif SAMPLE_FREQ == 16000
	r = F2_8000_OR_12000_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x1AC;
	wm8976_regs[WM8976_APPROX_SR] = 0x006;
#elif SAMPLE_FREQ == 32000
	r = F2_8000_OR_12000_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x16C;
	wm8976_regs[WM8976_APPROX_SR] = 0x002;
#elif SAMPLE_FREQ == 48000
	r = F2_8000_OR_12000_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x14C;
	wm8976_regs[WM8976_APPROX_SR] = 0x000;
#elif SAMPLE_FREQ == 12000
	r = F2_8000_OR_12000_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x1CC;
	wm8976_regs[WM8976_APPROX_SR] = 0x008;
#elif SAMPLE_FREQ == 24000
	r = F2_8000_OR_12000_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x18C;
	wm8976_regs[WM8976_APPROX_SR] = 0x004;
#elif SAMPLE_FREQ == 11025
	r = F2_11025_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x1CC;
	wm8976_regs[WM8976_APPROX_SR] = 0x008;
#elif SAMPLE_FREQ == 22050
	r = F2_11025_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x18C;
	wm8976_regs[WM8976_APPROX_SR] = 0x004;
#elif SAMPLE_FREQ == 44100
	r = F2_11025_FAMILY * FLOAT_MULTIPLYER;
	wm8976_regs[WM8976_CLK_CTRL]  = 0x14C;
	wm8976_regs[WM8976_APPROX_SR] = 0x000;
#else
#error Unsupported sample frequency
#endif

	do_div(r, dp_clk_freq);

	r_int = (unsigned int)r;
	plln = r_int / FLOAT_MULTIPLYER;
	r_rest = do_div(r, FLOAT_MULTIPLYER);
	k = 16777216 * r_rest;
	do_div(k, 10000ull);

	wm8976_regs[WM8976_PLL_N]   = plln;
	wm8976_regs[WM8976_PLL_K_1] = (k >> 18) & 0x03F;
	wm8976_regs[WM8976_PLL_K_2] = (k >>  9) & 0x1FF;
	wm8976_regs[WM8976_PLL_K_3] = k & 0x1FF;

#ifdef WOLFSON_IS_MASTER
	wm8976_regs[WM8976_CLK_CTRL] |= (1 << 0);
#else
	wm8976_regs[WM8976_CLK_CTRL] &= ~(1 << 0);
#endif

	printk("wm8976_set_pll(): pll_n = %u k = 0x%X r_int = %u; r_rest = %llu\n",
				plln, (int)k, r_int, r_rest);
}

static int wm_configure(struct i2c_client *client)
{
	struct snd_wm8976_partial_palette *pal = broadtile_music;
	int reg, ret, try;

	/* Configure PLL */
	wm8976_set_pll();

	/* Incorporate specific settings */
	for (pal = broadtile_music; pal->addr != WM8976_INVALID_REG_ADDR; pal++)
		wm8976_regs[pal->addr] = (wm8976_regs[pal->addr] & ~pal->mask) | pal->value;

	/* Tell the Wolfson... */
	for (reg = 1; reg < WM8976_NUM_REGS; reg++) {
		u8 buf[2];
		int data = wm8976_regs[reg];

		if (data == 0xffff)
			continue;

		buf[0] = (reg << 1) | ((data & 0x100) >> 8);
		buf[1] = data & 0xFF;

		try = 3;
	retry:
		if ((ret = i2c_master_send(client, buf, 2)) < 0) {
			/* Ease on the (shitty) i2c driver */
			msleep_interruptible(2 /* 1 */); // 1 mSec sleep
			if (try--)
				goto retry;

			printk("wm_configure(): i2c write failed at reg %d!\n", reg);
			return ret;
		}
	}

	return 0;
}


static unsigned long tdm_read(struct tdm *tdm, int reg)
{
	return readl(tdm->reg_base + reg);
}

static void tdm_write(struct tdm *tdm, int reg, unsigned long value)
{
	writel(value, tdm->reg_base + reg);
}

static void tdm_clk_disable(struct tdm *tdm)
{
	if (tdm->running) {
		clk_disable(tdm->tdm_clk);
		tdm->running = 0;
	}
}

static void tdm_clk_enable(struct tdm *tdm)
{
	if (!tdm->running) {
		clk_enable(tdm->tdm_clk);
		tdm->running = 1;
	}
}

static void tdm_set_reset(int enable)
{
	if (enable)
		reset_set(TDM_RESET);
	else
		reset_release(TDM_RESET);
}

static void tdm_reset(struct tdm *tdm)
{
	/* TDM clock mus be disabled when doing the ARM TDM SW-Reset. */
	tdm_clk_disable(tdm);

	/* Reset TDM register block */
	tdm_set_reset(1);
	udelay(10);
	tdm_set_reset(0);

	/* Enable TDM clock - for safe configuration of the HW machine internal
	 * states */
	tdm_clk_enable(tdm);
}


void tdm_setup(struct tdm *tdm)
{
	unsigned long val;
	long freq;

	freq = clk_round_rate(tdm->tdm_clk, 32 * SAMPLE_FREQ);
	if (freq < 0) {
		printk("clk_round_rate(tdm, %d): %ld\n", 32 * SAMPLE_FREQ, freq);
		return;
	}
	if (clk_set_rate(tdm->tdm_clk, freq)) {
		printk("clk_set_rate(tdm) failed!\n");
		return;
	}
	printk("TDM test: SCLK = %ldHz\n", freq);

	freq = clk_round_rate(tdm->dp_clk, DP_CLK_FREQ);
	if (freq < 0) {
		printk("clk_round_rate(dp, %d): %ld\n", DP_CLK_FREQ, freq);
		return;
	}
	if (clk_set_rate(tdm->dp_clk, freq)) {
		printk("clk_set_rate(dp) failed!\n");
		return;
	}
	if (clk_enable(tdm->dp_clk)) {
		printk("clk_enable(dp) failed!\n");
		return;
	}
	printk("TDM test: DP = %ldHz\n", freq);
	dp_clk_freq = freq;

	/*
	 * We should sw-reset the ARM TDM block perior to any TDM configuration
	 * change.
	 */
	tdm_reset(tdm);

	/* Configure TDM as master SYNC/TX/RX Edges, 2 time-slots, for
	 * "Left-Justified" TDM protocol flavour
	 */
	tdm_write(tdm, TDM_CONFIG_1,
#ifdef WOLFSON_IS_MASTER
		TDM_CONFIG_1_AUTO_SYNC |
#else
		TDM_CONFIG_1_FSYNC_MSTR_SLV |
		TDM_CONFIG_1_CLK_MSTR_SLV |
#endif
		TDM_CONFIG_1_SYNC_EDGE |
		TDM_CONFIG_1_TX_EDGE |
		0x0001 /* TDM_CONFIG_1_TIME_SLOTS: two time slots */
		);

	/* Configure MSB first, 64 banks, 2 fifo lines */
	tdm_write(tdm, TDM_CONFIG_2,
		TDM_CONFIG_2_SHIFT |
		0x0000 /* TDM_CONFIG_2_BANK_SIZE: 0 */
		);

	/* Configure Fsync signal parameters, according to TDM Format: */
#ifdef TDM_FORMAT_I2S
	/* Configure LRC Rise on bit 15 of First time-slot */
	val = 0x0F00;
#else
	/* Set for Left-Justified format: rise at bit 0, slot 0 */
	val = 0x0000;
#endif
	tdm_write(tdm, TDM_FSYNC, val);

	/* Configure Fsync High for 16 sclk cycles (16 bits) */
	tdm_write(tdm, TDM_FSYNC_DUR, 16-1);

	/* Configure TX & RX FIFOs Water-Mark */
	val = TDM_TX_FIFO_LEVEL | (TDM_RX_FIFO_LEVEL << 7);
	tdm_write(tdm, TDM_FIFO_WM, val);

	/* DATA_WIDTH = 16, CS_D = 0 (audio/voice data time slot), TX_EN = 1,
	 * RX_EN = 1 */
	tdm_write(tdm, 0x1800, 0x00D0); /* ENTRY = bank 0 */
	tdm_write(tdm, 0x1804, 0x01D0); /* ENTRY = bank 1 */
}


static void finish_out(int status, void *context)
{
	struct tdm *tdm = (struct tdm *)context;
	int ret;

	printk("TDM out list finished with status %d\n", status);
	if ((ret = dmw96dma_submit(tdm->dma_list_out)))
		printk("Resubmit failed with status %d\n", ret);
}

static void finish_in(int status, void *context)
{
	struct tdm *tdm = (struct tdm *)context;
	int ret;

	printk("TDM in list finished with status %d\n", status);
	if ((ret = dmw96dma_submit(tdm->dma_list_in)))
		printk("Resubmit failed with status %d\n", ret);
}


int tdm_start_output(struct tdm *tdm)
{
	int ret, len;
	u32 *buf;
	int int_en = 0;
	//u32 seed = 1234;

	tdm->buffer = kzalloc(BUFFER_SIZE*2, GFP_KERNEL);
	if (!tdm->buffer)
		return -ENOMEM;
	tdm->buffer_end = tdm->buffer + BUFFER_SIZE*2;
	tdm->rx_ptr = tdm->buffer;
	tdm->tx_ptr = tdm->buffer;

	/* fill with pattern */
	for (buf = tdm->buffer, len = BUFFER_SIZE/4; len > 0; len--) {
		//*buf = seed;
		//seed = 1664525*seed + 1013904223;
		*buf++ = ((u32)(sine[len&0xf])) << 12;
		*buf++ = ((u32)(sine[len&0xf])) << 12;
	}

	tdm->dma_list_out = dmw96dma_alloc_io_list(1, TDM_DMA_TX, &finish_out, tdm);
	if (!tdm->dma_list_out) {
		ret = -ENOMEM;
		goto free_buf;
	}

	if ((ret = dmw96dma_add_io_transfer_buf(tdm->dma_list_out, tdm->buffer,
		BUFFER_SIZE, NULL, NULL)))
		goto free_list;
	if ((ret = dmw96dma_add_io_transfer_buf(tdm->dma_list_out,
		tdm->buffer+BUFFER_SIZE, BUFFER_SIZE, NULL, NULL)))
		goto free_list;

	tdm->dma_list_in = dmw96dma_alloc_io_list(1, TDM_DMA_RX, &finish_in, tdm);
	if (!tdm->dma_list_in) {
		ret = -ENOMEM;
		goto free_list;
	}

	if ((ret = dmw96dma_add_io_transfer_buf(tdm->dma_list_in,
		tdm->buffer+BUFFER_SIZE, BUFFER_SIZE, NULL, NULL)))
		goto free_list2;
	if ((ret = dmw96dma_add_io_transfer_buf(tdm->dma_list_in, tdm->buffer,
		BUFFER_SIZE, NULL, NULL)))
		goto free_list2;

#ifdef OUTPUT_ENABLE
#ifdef OUTPUT_USE_DMA
	if ((ret = dmw96dma_submit(tdm->dma_list_out)))
		goto free_list2;
#else
	int_en = 0x0001;
#endif
#endif

#ifdef INPUT_ENABLE
#ifdef INPUT_USE_DMA
	if ((ret = dmw96dma_submit(tdm->dma_list_in)))
		goto free_list2;
#else
	int_en = 0x0004;
#endif
#endif

	/* Enable TDM */
	tdm_write(tdm, TDM_ENABLE, tdm_read(tdm, TDM_ENABLE) | 1);
	tdm_write(tdm, TDM_INT_EN, int_en);
	return 0;

free_list2:
	dmw96dma_free_list(tdm->dma_list_in);
free_list:
	dmw96dma_free_list(tdm->dma_list_out);
free_buf:
	kfree(tdm->buffer);
	return ret;
}

static irqreturn_t tdm_interrupt_handler(int irq, void *data)
{
	struct tdm *tdm = (struct tdm *)data;
	int pending;
	u32 *end = tdm->buffer_end;

	pending = tdm_read(tdm, TDM_INT_STATUS) & tdm_read(tdm, TDM_INT_EN);

	if (pending & 1) { /* TX */
		u32 *ptr = tdm->tx_ptr;

		while (tdm_read(tdm, TDM_STATUS) & 0x7f) {
			tdm_write(tdm, TDM_TX_FIFO, *ptr++);
			tdm_write(tdm, TDM_TX_FIFO, *ptr++);
			if (ptr >= end)
				ptr = tdm->buffer;
		}
		tdm->tx_ptr = ptr;
	}

	if (pending & 4) { /* RX */
		u32 *ptr = tdm->rx_ptr;

		while ((tdm_read(tdm, TDM_STATUS) >> 8) & 0x7f) {
			*ptr++ = tdm_read(tdm, TDM_RX_FIFO);
			*ptr++ = tdm_read(tdm, TDM_RX_FIFO);
			if (ptr >= end)
				ptr = tdm->buffer;
		}
		tdm->rx_ptr = ptr;
	}

	return IRQ_HANDLED;
}


static int wm8976_probe( struct i2c_client *client, const struct i2c_device_id *id )
{
	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		printk("wm8976_probe: missing functionality\n");
		return -ENODEV;
	}

	wm_configure(client);

	printk("tdm-test: WM8976 initialized\n");

	return 0;
}


static int wm8976_remove( struct i2c_client *client )
{
	return 0;
}

static const struct i2c_device_id wm8976_id[] = {
	{ "wm8976"  },
	{ }
};

static struct i2c_driver i2c_driver_8976 = {
	.driver = {
		.name = "wm8976",
	},
	.probe	= wm8976_probe,
	.remove = wm8976_remove,
	.id_table = wm8976_id,
};


int tdm_init(void)
{
	int ret;
	struct tdm *tdm;

	tdm = kzalloc(sizeof(*tdm), GFP_KERNEL);

	tdm->reg_base = ioremap_nocache(TDM_REG_BASE, TDM_REG_SIZE);
	if (!tdm->reg_base) {
		printk("wm8976: ioremap_nocache() failed for 96-TDM on FPGA\n");
		return -1;
	}

	tdm->dp_clk = clk_get_sys("soc-audio", NULL);
	if (IS_ERR(tdm->dp_clk)) {
		ret = PTR_ERR(tdm->dp_clk);
		printk("wm8976: DP clock get failed\n");
		goto err_ioremap;
	}

	tdm->tdm_clk = clk_get_sys(TDM_CLOCK, NULL);
	if (IS_ERR(tdm->dp_clk)) {
		ret = PTR_ERR(tdm->tdm_clk);
		printk("wm8976: TDM clock get failed\n");
		goto err_clk;
	}

	tdm_setup(tdm);

	if (i2c_add_driver(&i2c_driver_8976))
		printk("wm8976: register failed!\n");

	ret = request_irq(TDM_IID, tdm_interrupt_handler, IRQF_DISABLED, "TDM", tdm);
	if (ret) {
		printk("TDM: could not get irq: %d\n", ret);
		goto err_irq;
	}

	if ((ret = tdm_start_output(tdm))) {
		printk("tdm_start_output: failed with %d\n", ret);
		goto err_irq;
	}

	printk("tdm_init: done\n");
	return 0;

err_irq:
	clk_put(tdm->tdm_clk);
err_clk:
	clk_put(tdm->dp_clk);
err_ioremap:
	iounmap(tdm->reg_base);
	return ret;
}

late_initcall(tdm_init);

