/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* EVENT.C --- Produce events for the rest of the system. */

#ifndef NO_ASM
#pragma inline
#endif

#include <stdio.h>
#include <assert.h>
#include <setjmp.h>
#include <bios.h>
#include <time.h>
#include <mem.h>
#include <graphics.h>
#include "window.h"
#include "wintern.h"
#include "key.h"

enum {

repeat_wait = delta(.45),               /* number of ticks before repeat begins */
repeat_tick = delta(.1),                /* seconds between repeat events */

};
int double_click_delta = delta(.5);     /* clock ticks twixt double clicks */

int ptr_x, ptr_y;                       /* current pointer location */
unsigned button_mask;                   /* current button status */
unsigned ctrl_key_mask;                 /* current control key status */

#define MAX_EVENTS 500                  /* max events that can be buffered */
static EVENT_REC events[MAX_EVENTS];    /* event buffer */
static int e_head = 0, e_tail = 0;      /* head and tail pointers */
static unsigned long clk;               /* last value of system clock */
static WINDOW grabbing_window = NULL;   /* window grabbing mouse events */
static int grab_button;                 /* button pressed to get grab */


WINDOW root_window;                     /* ancestor of all windows */
WINDOW ptr_window;                      /* window pointer currently inhabits */
WINDOW focus_window;                    /* window with the keyboard focus */
WINDOW pending_lose_focus;              /* window waiting to get a lose focus. */
static int protect_focus_p;             /* remember focus if implicitly unset */
static WINDOW protected_focus;          /* focus to be restored by unprotect */
static WINDOW hotkey_root_stk[MAX_N_ROOTS];     /* hotkey root stack */
static int hrsp;                        /* hotkey root stack pointer */

static struct timer {
  unsigned long delta;
  unsigned long end_clk;
  int next_ticker;
  short reset_p;
} timers[N_TIMERS];                     /* user timers for tick events */
static int tickers = -1;                /* queue of timers waiting tick */

HANDLER_CODE global_event_handler[eN];  /* global event handlers */

/* ----- Timers ----------------------------------------------------- */

/* Turn off a timer. */
void stop_timer (int n)
{
  int p, q;

  assert(n < N_TIMERS);

  if (timers[n].delta == 0)
    return;

  for (q = -1, p = tickers; p != n; q = p, p = timers[p].next_ticker)
    assert(p != -1);

  if (q == -1)
    tickers = timers[p].next_ticker;
  else
    timers[q].next_ticker = timers[n].next_ticker;
  memset(&timers[n], 0, sizeof(struct timer));
}

/* Put a timer on the list of tickers in
   increasing order of end clock value. */
LOCAL(void) enqueue_timer(int n)
{
  int p, q;
  for (q = -1, p = tickers; p != -1; q = p, p = timers[p].next_ticker)
    if (timers[p].end_clk > timers[n].end_clk)
      break;
  if (q == -1) {
    timers[n].next_ticker = tickers;
    tickers = n;
  }
  else {
    timers[n].next_ticker = p;
    timers[q].next_ticker = n;
  }
}

/* Set/reset a timer to generate tick events. */
void start_timer (int n, unsigned long delta, short reset_p)
{
  struct timer *t = &timers[n];
  assert(n < N_TIMERS);
  if (t->delta > 0)
    stop_timer(n);
  t->delta = delta;
  t->end_clk = delta + clk;
  t->reset_p = reset_p;
  enqueue_timer(n);
}

/* ----- Repeated actions ------------------------------------------- */

static ACTION_CLOSURE_REC repeat_action;
static int n_repeat_ticks;
#pragma argsused
static void do_repeat(EVENT e)
{
  if (n_repeat_ticks < repeat_wait)
    ++n_repeat_ticks;
  else
    (*repeat_action.code)(repeat_action.env);
}

/* Set the handler for button style hold-to-repeat stuff
   and perform the initial action. */
void begin_repeat(ACTION_CODE code, ENV env)
{
  assert(global_event_handler[eTICK(tREPEAT)] == null_handler_code);
  global_event_handler[eTICK(tREPEAT)] = do_repeat;
  repeat_action.code = code;
  repeat_action.env = env;
  n_repeat_ticks = 0;
  start_timer(tREPEAT, repeat_tick, 1);
  (*code)(env);
}

/* Stop the current repeat action. */
void end_repeat(void)
{
  global_event_handler[eTICK(tREPEAT)] = null_handler_code;
  stop_timer(tREPEAT);
}

/* ----- Double click actions --------------------------------------- */

/* Pointer given by last button click. */
static void *last_click = NULL;

/* Reset last_click when timer expires. */
#pragma argsused
static void unclick_handler(EVENT e) { last_click = NULL; }

int double_click_p(void *p)
{
  if (p != last_click) {
    last_click = p;
    start_timer(tDOUBLE_CLICK, double_click_delta, 0);
    return 0;
  }
  else {
    stop_timer(tDOUBLE_CLICK);
    last_click = NULL;
    return 1;
  }
}

/* ----- Event generation ------------------------------------------- */

/* Allocate the next event in the circular queue. */
LOCAL(EVENT) ealloc(EVENT_TYPE type)
{
  EVENT e = &events[e_head];
  if (++e_head >= MAX_EVENTS)
    e_head = 0;
  assert(e_head != e_tail);
  e->type = type;
  return e;
}

/* Add an entry/departure event to the queue. */
LOCAL(void) push_entry_event(EVENT_TYPE type,
			     WINDOW window,
			     int x, int y,
			     unsigned mask)
{
  if (window->event_mask & bit(type)) {
    EVENT_REC e;
    e.type = type;
    e.mouse.window = window;
    e.mouse.mask = mask;
    e.mouse.x = x - window->x;
    e.mouse.y = y - window->y;
    (*window->handler)(&e);
  }
}


/* Add a map event to the queue.
   Called by window.c when a window is mapped. */
void push_map_event(EVENT_TYPE type, WINDOW window)
{
  if (window->event_mask & bit(type)) {
    EVENT_REC e;
    e.type = type;
    e.map.window = window;
    (*window->handler)(&e);
  }
}

/* Return the window where a mouse event has occurred. 
   Update global state for current pointer window. */
LOCAL(WINDOW) mouse_event_window(int x, int y, EVENT_TYPE type, unsigned mask)
{
  WINDOW w, pw;

  /* Find window of event type and also window pointer lies in. */
  get_mouse_event_window(x, y, type, &w, &pw);

  if (type == eMOUSE_MOTION && pw != ptr_window) {
    push_entry_event(eDEPARTURE, ptr_window, x, y, mask);
    ptr_window = pw;
    /*
    { char buf[80];
      int color = getcolor();
      sprintf(buf, "%d mew type %d", ptr_window->id, type);
      setcolor(cBLUE);
      outtextxy(200, 200, "\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb");
      setcolor(cWHITE);
      outtextxy(200, 200, buf);
      setcolor(color);
    }
    */
    push_entry_event(eENTRY, ptr_window, x, y, mask);
  }
  return w;
}

/* Simulate null mouse motion to force departure/entry
   events for state changes.

   For maps and unmaps, we must send the departure event
   before changing the screen, then the entry event after.
   This is a special gronge for xor cursors that use
   entries and departures and must stay in synch when dialogs
   and pulldowns are mapped on top of the cursor window. */
WINDOW begin_update_ptr_window(void)
{
  WINDOW pw, w;

  if (grabbing_window == NULL) {

    get_mouse_event_window(ptr_x, ptr_y, eMOUSE_MOTION, &w, &pw);

    if (ptr_window != NULL && pw != ptr_window) {
      push_entry_event(eDEPARTURE, ptr_window, ptr_x, ptr_y, button_mask);
      /* Pointer window is now in limbo. */
      ptr_window = NULL;
      return pw;
    }
  }
  return NULL;
}


void end_update_ptr_window(WINDOW pw)
{
  if (pw != NULL) {
    ptr_window = pw;
    /*
    { char buf[80];
      int color = getcolor();
      sprintf(buf, "%d update", ptr_window->id);
      setcolor(cBLUE);
      outtextxy(200, 200, "\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb");
      setcolor(cWHITE);
      outtextxy(200, 200, buf);
      setcolor(color);
    }
    */
    push_entry_event(eENTRY, ptr_window, ptr_x, ptr_y, button_mask);
  }
}

void update_ptr_window(void)
{
  end_update_ptr_window(begin_update_ptr_window());
}

/* Add a mouse event to the queue. */
LOCAL(void) push_mouse_event(EVENT_TYPE type, int x, int y, int mask, MOUSE_BUTTON button)
{
  WINDOW w;
  EVENT e;

  /* See if any window has pointer grabbed. */
  if (grabbing_window == NULL) {

    /* Find window of event type and also window pointer lies in. */
    w = mouse_event_window(x, y, type, mask);

    /* If no event window was found, we're done. */
    if (w == NULL)
      return;

    /* If event was a button press, initiate grab. */
    if (type == eBUTTON_PRESS) {
      grabbing_window = w;
      grab_button = button;
    }
  }
  else {

    /* Event window is automatically grab window. */
    w = grabbing_window;

    /* If this was release of the grabbing button, terminate grab. */
    if (type == eBUTTON_RELEASE && button == grab_button)
      ungrab_mouse();

    /* If requested event type was not requested, we done. */
    if (!(w->event_mask & bit(type)))
      return;
  }

  /* Allocate event on queue and fill it in. */
  e = ealloc(type);
  e->mouse.window = w;
  e->mouse.x = x - w->x;
  e->mouse.y = y - w->y;
  e->mouse.mask = mask;
  e->mouse.button = button;
}

/* Release the grab on the pointer. */
void ungrab_mouse(void)
{
  if (grabbing_window != NULL) {
    grabbing_window = NULL;
    update_ptr_window();
  }
}

LOCAL(void) push_focus_event(EVENT_TYPE type, WINDOW window, WINDOW other)
{
  if (window->event_mask & bit(type)) {
    EVENT_REC e;
    e.type = type;
    e.focus.window = window;
    e.focus.other_window = other;
    (*window->handler)(&e);
  }
}

static_WINDOW(dummy);


/* Cause lose and gain focus events due to focus change. 
   This has to be reentrant. Focus events can call set_focus
   and can restart event processing, e.g. by calling
   run_modal_dialog to display an error message due to
   a change focus validation.

   Therefore this "pump" routine is called by both set_focus 
   and by next_event.  The idea is that every recursive entry 
   checks to see if there are events that need to be sent and 
   sends them as soon as logically consistent. (See invariants 
   below.) If no work to do, then just leave.

   Thus any crazy nesting of calls to set_focus and next_event
   must result in a smooth stream of events:  lose(A,B) gain(B,A)
   lose(B,C) gain(C,B)...  It _is_ possible that a call set_focus(Z)
   will never cause gain(Z,.) lose(Z,.) to occur, but if the
   gain(Z,.) _does_ occur, then a lose(Z,.) will also.

   We do this by keeping track of the state of the event stream
   with three static variables:  focus_window, pending_lose_focus,
   and last_loser.

   Axioms and invariants:
     0.  Start of push_focus_event constitutes the event. (Don't
	 care when push_focus_event finally returns!)
     1.  focus_window contains most recent request to change focus.
     2.  pending_lose_focus contains the window last sent a gain focus
	 and has not yet been sent lose focus.
     3.  last_loser contains the last window to be sent a lose event.
*/
LOCAL(void) pump_focus_events(void)
{
  /* Loop while event sending is not "caught up."
     This is when gain-focus hasn't been sent 
     for current focus window yet, i.e. while 
     focus window isn't yet pending a lose event. */
  while (pending_lose_focus != focus_window) {

    static WINDOW last_loser;

    /* If there's a window that needs a lose focus...*/
    if (pending_lose_focus) {

      /* Push lose focus event to pending window.  Since we're doing 
	 that, have to set last_loser to maintain the invariants. */
      last_loser = pending_lose_focus;

      /* Also, the pending window will no longer be pending,
	 so set pending_lose_focus to 0 to maintain invariants. */
      pending_lose_focus = 0;
      push_focus_event(eLOSE_FOCUS, last_loser, focus_window);
    }
	
    /* See if a re-entry has caught us up (while we were
       doing the lose-focus above). */
    if (pending_lose_focus != focus_window) {

      /* No.  We have to do it. Since we're sending gain-focus, 
	 that window is now pending a corresponding lose. Set
	 pending_lose_focus to maintain the invariants. */
      pending_lose_focus = focus_window;
      push_focus_event(eGAIN_FOCUS, focus_window, last_loser);
    }
    /* Have to loop to try again because processing gain-focus 
       could have put us behind again (e.g. by calling set_focus). */
  }
}

void set_focus(WINDOW new_focus_window)
{
  if (new_focus_window == NULL) 
    new_focus_window = dummy;
  if (w_status_p(new_focus_window, wsVISIBLE)) {
    focus_window = new_focus_window;
    pump_focus_events();
  }
}

void protect_focus(void)
{
  if (!protect_focus_p) {
    protect_focus_p = 1;
    protected_focus = NULL;
  }
}

void unprotect_focus(void)
{
  if (protected_focus != NULL)
    set_focus(protected_focus);
  protect_focus_p = 0;
}

/* Surrender the focus to the nearest ancestor
   with keystroke or gain focus events enabled. */
void unset_focus(WINDOW w)
{
  if (w == focus_window) 
    if (protect_focus_p) {
      set_focus(NULL);
      protected_focus = w;
    }
    else
      for (;;) {
	if (w == root_window) {
	  set_focus(NULL);
	  break;
	}
	w = w->parent;
	if (w_status_p(w, wsVISIBLE) && 
	    w->event_mask & (bit(eKEYSTROKE)|bit(eGAIN_FOCUS))) {
	  set_focus(w);
	  break;
	}
      }
}

void push_hotkey_root(WINDOW w)
{
  assert(hrsp < MAX_N_ROOTS);
  hotkey_root_stk[hrsp++] = w;
}

void pop_hotkey_root(void)
{
  assert(hrsp > 0);
  --hrsp;
}

/* Send hot key events to all windows with mask bit set. 
   If both VISIBLE_HOTKEY and HOTKEY bits are set, send
   HOTKEY only if window is not visible. So a window
   never gets more than one hotkey event per keystroke. 
   No hotkey events are sent if the pointer is grabbed. */
LOCAL(void) push_hotkey_events(int key, WINDOW w)
{
  EVENT e;

  if (grabbing_window != NULL)
    return;

  if (event_enabled_p(w, eVISIBLE_HOTKEY) && w_status_p(w, wsVISIBLE)) {
    e = ealloc(eVISIBLE_HOTKEY);
    e->hotkey.window = w;
    e->hotkey.key = key;
  }
  else if (event_enabled_p(w, eHOTKEY)) {
    e = ealloc(eHOTKEY);
    e->hotkey.window = w;
    e->hotkey.key = key;
  }
  if (w->kids != NULL) {
    WINDOW k = w->kids;
    do {
      push_hotkey_events(key, k);
    } while ((k = k->next) != w->kids);
  }
}

/* Add a control key press/release to the queue. */
LOCAL(void) push_ctrl_key_event(EVENT_TYPE type, unsigned mask, int key)
{
  EVENT e = ealloc(type);
  e->type = type;
  e->ctrl_key.key = key;
  e->ctrl_key.mask = mask;
}


/* Adjust the event generator's state due 
   or unmapping of a window.  Various things can change. */
void update_state_for_unmap(void)
{
  /* If the grabbing window is now invisible, ungrab. */
  if (grabbing_window != NULL && !w_status_p(grabbing_window, wsVISIBLE))
    ungrab_mouse();

  /* Ensure roots in root stack are still visible. */
  check_roots_visible();

  /* Ensure focus is visible. */
  if (!w_status_p(focus_window, wsVISIBLE))
    unset_focus(focus_window);
}

#ifdef LOCAL_CLOCK
#include <sys\timeb.h>

#if 10*(long)CLOCKS_PER_SEC != (long)(10 * CLOCKS_PER_SEC)
#error Bad integer arithmetic in local clock.
#endif

static struct timeb start[1];

static time_t clock(void)
{
  struct timeb now[1];

  if (!start->time)
    ftime(start);

  ftime(now);
  return (now->time - start->time) * (time_t)CLOCKS_PER_SEC +
	 (now->millitm - start->millitm);
}

#endif

/* Return the next event, whatever it is. */
LOCAL(EVENT) next_event(void)
{
  int x, y, i, bit;
  unsigned mask;
  EVENT e;

  /* Pump events to modal dialogs, etc. */
  pump_focus_events();

  while (e_head == e_tail) {

    get_mouse_status(&x, &y, &mask);

    /* Generate mouse motion events. */
    if (x != ptr_x || y != ptr_y) {
      ptr_x = x;
      ptr_y = y;
      push_mouse_event(eMOUSE_MOTION, x, y, mask, bNONE);
    }

    /* Generate mouse button events. */
    if (mask != button_mask) {
      for (i = 0, bit = 1; i < bN; ++i, bit <<= 1) {
	unsigned new_bit = mask & bit;
	unsigned old_bit = button_mask & bit;
	if (new_bit > old_bit)
	  push_mouse_event(eBUTTON_PRESS, x, y, mask, i);
	else if (new_bit < old_bit)
	  push_mouse_event(eBUTTON_RELEASE, x, y, mask, i);
      }
      button_mask = mask;
    }

    /* Generate control key events. */
    mask = bioskey(2);
    if (mask != ctrl_key_mask) {
      for (i = 0, bit = 1; i < ckN; ++i, bit <<= 1) {
	unsigned new_bit = mask & bit;
	unsigned old_bit = ctrl_key_mask & bit;
	if (new_bit > old_bit)
	  push_ctrl_key_event(eCTRL_KEY_PRESS, mask, i);
	else if (new_bit < old_bit)
	  push_ctrl_key_event(eCTRL_KEY_RELEASE, mask, i);
      }
      ctrl_key_mask = mask;
    }

    /* Generate keystroke events. */
    if (bioskey(1)) {
      int key = bioskey(0);
      if (key & 0xff)
	key &= 0xff;
      if ((ctrl_key_mask & bit(ckALT)) && key == ' ') 
	key = ALT_SPACE;
      if (focus_window->event_mask & bit(eKEYSTROKE)) {
	e = ealloc(eKEYSTROKE);
	e->keystroke.window = focus_window;
	e->keystroke.key = key;
	e->keystroke.mask = ctrl_key_mask;
      }
      else
	push_hotkey_events(key, hrsp ? hotkey_root_stk[hrsp - 1] : root_window);
    }

    /* Generate timer events. */
    clk = clock();
    while (tickers != -1 && clk >= timers[tickers].end_clk) {
      int n = tickers;
      struct timer *t = &timers[n];
      e = ealloc(eTICK0 + n);
      e->tick.timer = n;
      tickers = t->next_ticker;
      if (t->reset_p) {
	t->end_clk = clk + t->delta;
	enqueue_timer(n);
      }
      else
	memset(t, 0, sizeof(struct timer));
    }
  }

  e = &events[e_tail];
  if (++e_tail >= MAX_EVENTS)
    e_tail = 0;
  return(e);
}

void refuse_keystroke(EVENT e)
{
  int key = e->keystroke.key;
  assert(e->type == eKEYSTROKE);
  push_hotkey_events(key, hrsp ? hotkey_root_stk[hrsp - 1] : root_window);
}

/* Fetch events and handle them.  
   Note: The global handlers are _not_ currently 
   called on per-window events. */
LOCAL(void) handle_events(void)
{
  for (;;) {
    EVENT e = next_event();
    EVENT_TYPE t = e->type;
    if (t < eN_PER_WINDOW_EVENT_TYPES)
      (*e->mouse.window->handler)(e);
    else 
      (*global_event_handler[t])(e);
  }
}

/* Initiate a run and handle stops. (Re-entrant.) */

#define MAX_RUN_NESTING 8

#ifdef NO_ASM

static jmp_buf run_stack[MAX_RUN_NESTING];
static int rsp;

#define HIBIT ((int)~(~0u >> 1))

int run()
{
  int result;

  assert(rsp < MAX_RUN_NESTING);
  result = setjmp(run_stack[rsp++]);
  if (result == 0) {
	handle_events();
	return -1;
  }
  return result & ~HIBIT;
}

void stop(int n)
{
  assert(rsp > 0);
  longjmp(run_stack[--rsp], n | HIBIT);
}

#ifdef NOT_USED

void stopall(void)
{
  rsp = 0;
  longjmp(&run_stack[0]);
}

#endif

#else

static struct { int sp, bp; } run_stack[MAX_RUN_NESTING];
static int rsp;

/* Initiate a run and handle stops. (Re-entrant.)
   This could be done with longjmp, but this is
   much faster and smaller. */

#pragma argsused

int _run(int dummy)
{
  asm {
    mov si, word ptr DGROUP:rsp
    mov di, si
    shl si, 1
    shl si, 1
    mov word ptr DGROUP:run_stack[si], sp
    mov word ptr DGROUP:run_stack[si+2], bp
    inc di
    mov word ptr DGROUP:rsp, di
  }
  handle_events();
  return _AX;
}

void stop(int n)
{
  asm {
    mov si, word ptr DGROUP:rsp
    dec si
    mov di, si
    shl si, 1
    shl si, 1
    mov ax, n
    mov sp, word ptr DGROUP:run_stack[si]
    mov bp, word ptr DGROUP:run_stack[si+2]
    mov word ptr DGROUP:rsp, di
  }
}


#ifdef NOT_USED

void stopall(void)
{
  asm {
    mov si, di
    mov sp, word ptr DGROUP:run_stack
    mov bp, word ptr DGROUP:run_stack+2
  }
}

#endif

#endif

/* ----- Utility functions ------------------------------------------ */

/* Generalized press and release include
   entry/depart with button pressed. */
int generalized_press_p(EVENT e, int button)
{
  return e->type == eBUTTON_PRESS && e->mouse.button == button ||
	 e->type == eENTRY && (e->mouse.mask & bit(button));
}

int generalized_release_p(EVENT e, int button)
{
  return e->type == eBUTTON_RELEASE && e->mouse.button == button ||
	 e->type == eDEPARTURE && (e->mouse.mask & bit(button));
}

/* Trivial event handler. */
void do_nothing(void) { }

/* ----- Initialization --------------------------------------------- */

/* Provide minimum information to start events flowing. */
int init_events(int x0, int y0, int x1, int y1)
{
  int i;

  if (!set_mouse_arena(x0, y0, x1, y1))
    return(0);
  get_mouse_status(&ptr_x, &ptr_y, &button_mask);
  ctrl_key_mask = bioskey(2);
  show_mouse_cursor(1);
  ptr_window = root_window = make_root_window(x0, y0, x1, y1);
  open_window(dummy, root_window, 0, 0, 1, 1, 0, 
	      TRANSPARENT, TRANSPARENT, bit(eGAIN_FOCUS));
  map_window(dummy);
  pending_lose_focus = focus_window = dummy;
  for (i = 0; i < eN; ++i)
    global_event_handler[i] = null_handler_code;
  global_event_handler[eTICK(tDOUBLE_CLICK)] = unclick_handler;
  return(1);
}
