/*
 * SOURCE:  help.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: help engine
 *
 */
#include "main.h"

static struct Gadget *helpgadhandler( struct Gadget *, struct Gadget *,
    unsigned, unsigned, unsigned long );
static char *_printline( char *, unsigned, unsigned, int );

extern char *HlpName;
extern int int_pending;
extern struct IntuitionBase *IntuitionBase;
extern struct Editor ed;
extern BOOL _remark;

static void *topic_stack = NULL;

void PushTopic( int topic ) {

    topic_stack = PushStack( topic_stack, (APTR)&topic, sizeof(int) );
}

int PopTopic( void ) {

    int topic = 0;

    if( topic_stack ) {
        topic = *(int *)topic_stack;
        topic_stack = PopStack( topic_stack );
    }
    return topic;
}

int GetTopic( void ) {

    if( topic_stack ) return *(int *)topic_stack;
    return -1;
}

void ClearTopicStack( void ) {

    FreeStack( topic_stack );
    topic_stack = NULL;
}


static char *topics[] = {
"DEFAULT",
"KEYS",
"MOUSE",
"BOXES",
"BOXES_KEY",
"BOXES_MOUSE",
"ALLGEMEIN",
"READONLY",
"UNASSIGNED",
"LINETOOLONG",
"NOCREAT",
"INVARG",
"NOALTBUF",
"NOSAVCBOARD",
"NOLOADCBOARD",
"NOPASTE",
"PARTOOLARGE",
"PARERROR1",
"PUTLINE",
"FILETOOLARGE",
"RECOVER",
"REFRESH",
"OVERWRITE",
"DISKFULL",
"SAVVIRT1",
"SAVVIRT2",
"MODIFIED",
"SETVMODE",
"GOTO",
"ABOUT",
"MEMORY",
"PACK",
"NOSLOTS",
"SETTINGS",
"COLORSETUP",
"DOSREQUEST",
"DOSREQOPTS",
"SAVEREQ",
"CREATEREQ",
"SETFILE",
"FILEINFO",
"ADDFILE",
"VISIT",
"SAVEFILE",
"MERGE",
"SETFILE2",
"LAUNCH1",
"LAUNCH2",
"SEARCH",
"SRCFILES",
"CONFEMPTY",
"QUERY",
"PRINT",
"PAGES",
"COMBINE",
"LATEXCMD",
"PREVIEWCMD",
"VIDEOMODE",
"PRINTCMD",
"DRIVER",
"EXLIB1",
"EXLIB2",
"CMDOPTS",
"RESPONSE",
"RESPONSE2",
"DOSCMD",
"CUSTOMMENU",
"CUSTOM2",
"ENVIRONMENT",
"PSEUDOFILES",
"VIEWERR",
"PROJDIR",
"PROJDIR2",
"FORMAT",
"FILES",
"TEXENV",
"MENU",
"CHANGEINI",
"FILESYSTEM",
"CLIPBOARD",
"UNDO",
"CMDLINEERR",
};

#define WIN_WIDTH  65
#define WIN_HEIGHT 21
/* This structure is accessed by help() and helpgadhandler() to
   access the internal topic stack (its) */
struct TopicStruct {
    unsigned offset; /* offset in reloc table for this topic */
    int cursorx;     /* cursor position for this topic */
    int cursory;
    int windowy;
};
static struct TopicStruct topic;
static void *its = NULL;  /* internal topic stack */
unsigned char _far *tbuf; /* file buffer for this topic */

unsigned int help( int label ) {

    struct MarkRegion mr = { 0,0,0,0,"",MR_INITVAL};
    int i;
    struct IntuitionBase savibase;

    savibase = *IntuitionBase;
    IntuitionBase->WindowColor = CR_WINDOW;
    IntuitionBase->GadgetColor = CR_BOOLGADGET;
    IntuitionBase->GadgetSelect = CR_BOOLGADSELECT;
    IntuitionBase->GadgetHot = CR_GADHOTKEY;
    IntuitionBase->GadgetHotSelect = CR_GADSELHOT;
    IntuitionBase->BorderColor = CR_BORDER;


    /* Reset the topic stack */
    if (its) {
        FreeStack( its );
        its = NULL;
    }

    if( label == -1 ) {
        /* look if something is marked */
        mr = *(m_getmr());

        if( mr.flags & MR_MARKED && !(mr.flags & MR_MULTILINE) ) {
            /* Push this topic onto the stack */
            aschelp( mr.buffer );
            CLEARBIT( ed.Flags, MARKED );
            _remark = TRUE;
            *IntuitionBase = savibase;
            return 0;
        }

        label = GetTopic();
        if( label == -1 )
            label = HLP_DEFAULT;
    }

    aschelp( topics[label] );
    *IntuitionBase = savibase;
}


void lasthelp( void ) {

    struct IntuitionBase savibase;

    savibase = *IntuitionBase;
    IntuitionBase->WindowColor = CR_WINDOW;
    IntuitionBase->GadgetColor = CR_BOOLGADGET;
    IntuitionBase->GadgetSelect = CR_BOOLGADSELECT;
    IntuitionBase->GadgetHot = CR_GADHOTKEY;
    IntuitionBase->GadgetHotSelect = CR_GADSELHOT;
    IntuitionBase->BorderColor = CR_BORDER;

    /* invoke the help engine directly */
    aschelp("");

    *IntuitionBase = savibase;
}


void aschelp( char *topicname ) {

    static int recur = 0;

    static struct RelocTable *rtab_first;
    static unsigned rtab_num;

    static struct Window win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, NULL, 0L
    };

    static struct IntuiText it_back = {
        NULL, NULL, "&Back", 1, 0
    };
    static struct Gadget g_back = {
        NULL, &it_back, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText it_contents = {
        NULL, NULL, "&Contents", 1, 0
    };
    static struct Gadget g_contents = {
        &g_back, &it_contents, (WIN_WIDTH-12)/2, WIN_HEIGHT-4,
        12, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText it_ok = {
        NULL, NULL, "OK", 1, 0
    };
    static struct Gadget g_ok = {
        &g_contents, &it_ok, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET,
    };

    static struct Request req = {
        NULL, NULL, NOBARHILITE|OWNREDRAW };

    static struct Gadget gad_req = {
        &g_ok, NULL, 2, 1, 62, 16, REQGADGET | ACTIVATED,
        (APTR)&req,
    };

    int fptr = -1;
    unsigned char *cp;
    int i, j;
    unsigned char linelen;
    unsigned long offset, size, tsize, scan;
    char title[SCREEN_WIDTH];
    struct Gadget *g;
    int newlabel;
    unsigned long *freq_buf = NULL;
    unsigned char _far *cbuf = NULL;

    unsigned int ( *savint )();

    struct RelocTable *rtab;
    struct Cursor sav;

    BOOL winopen = FALSE;

    /* avoid execution during interrupts */
    if( int_pending )
        return;

    /* avoid recursion */
    if( recur >= 1 )
        return;

    ++recur;

    /* clear hook verctor */
    savint = SetHook( NULL, HOOK_WAIT );

    GetCursor( -1, &sav );

    win.Title = NULL;
    if( OpenWindow( &win, 0 ) )
        winopen = TRUE;
    else
        goto done;

    /* try to load help file */
    fptr = Open( HlpName, ACCESS_READ );
    if( fptr == -1 ) {
        PutError( -1, HlpName );
        goto done;
    }

    _Read( fptr, (char _far *)&size, sizeof(unsigned long) );
    if( size != HELP_MAGIC ) {
        DoErrorBox( HLP_DEFAULT, "Help File Corrupt:\n'%s'", HlpName );
        goto done;
    }

    if( (freq_buf = AllocHeap( COUNTTABSIZE )) == NULL ) goto done;
    _Read( fptr, (char _far *)freq_buf, 4 * 256 );

    /* Remember: by calling PutTree(), compress gains control over
                 the frequency table. The calling rountine must
                 not call free() any more */


    PutTree( freq_buf );

    if( BuildTree() != CE_OK ) {
        DoErrorBox( HLP_DEFAULT, "Unable to expand database" );
        goto done;
    }

    _Read( fptr, (char _far *)&size, sizeof(unsigned long) );
    if( (rtab_first = AllocHeap( (unsigned)size )) == NULL ) goto done;

    _Read( fptr, (char _far *)&rtab_num, sizeof(unsigned) );
    _Read( fptr, (char _far *)rtab_first, (unsigned)size );

    tbuf = NULL;

    if (*topicname || (its==NULL) ) {

        topic.offset = 0;

        /* search in reloc table */
        for( rtab = rtab_first, i = 0; i < rtab_num; ++i ) {
            if( strcmp( topicname, rtab->name ) == 0 ) {
                /* topic found */
                topic.offset = i;
                break;
            }
            /* get next entry in reloc table */
            (unsigned char *)rtab += sizeof(unsigned long) + strlen(rtab->name)
                                     + sizeof(char);
        }
        if( i == rtab_num ) {
            DoErrorBox( HLP_DEFAULT, "Help on topic not found:\n'%s'", topicname );
        }

        /* reinitialize cursor position */
        topic.cursorx = -1;

    } else {
        topic = *(struct TopicStruct *)its;
        its = PopStack( its );
    }

    do {
        /* serach for offset */
        for( rtab = rtab_first, i = 0; i < rtab_num; ++i ) {
            if( i == topic.offset ) {
                /* topic found */
                break;
            }
            (unsigned char *)rtab += sizeof(unsigned long) + strlen(rtab->name)
                                  + sizeof(char);
        }

        Seek( fptr, rtab->offset, SEEK_START );
        /* read lines */
        _Read( fptr, (char _far *)&tsize, sizeof(unsigned long) );
        if( tbuf ) free( tbuf );
        tbuf = AllocHeap( (unsigned)tsize );
        if( tbuf == NULL ) goto done;

        _Read( fptr, (char _far *)tbuf, sizeof(unsigned) );
        req.NumIt = *(unsigned _far *)tbuf;
        if( req.NumIt > 0 ) --req.NumIt;

        /* read length of compressed entry */
        _Read( fptr, (char _far *)&size, sizeof(unsigned long) );
        cbuf = AllocHeap( (unsigned)(size/8+1) );
        if( cbuf == NULL ) goto done;
        _Read( fptr, cbuf, (unsigned)(size/8+1) );
        size = Decode( tbuf, tsize, cbuf, size );
        free( cbuf );

        strcpy( title, "Help:" );
        strcat( title, tbuf+sizeof(unsigned) );
        win.Title = title;
        RefreshWindow( &win );

        /* check stack for entry */
        if( its )
            CLEARBIT( g_back.Flags, DISABLED );
        else
            SETBIT( g_back.Flags, DISABLED );

        /* if this topic is accessed for the first time, go to first button
           by calling helpgadhandler() */
        if( topic.cursorx == -1 ) {
            topic.cursorx = 0;
            req.CurrIt = 0;
            req.WinTop = 0;
            IntuitionBase->InputEvent->key = BK_TAB;
            helpgadhandler( &gad_req, &gad_req, win.Left, win.Top,
                            CG_OWNREDRAW|CG_CHANGE );
        }

        g = OpenGadget( &gad_req, win.Left, win.Top, helpgadhandler );

        if( g == &gad_req || g == &g_contents ) {
            /* change topic ? */
            if( g == &g_contents ) {
                topic.offset = 0;
                req.CurrIt = req.WinTop = 0;
            }
            /* the topic is pushed on the stack by heplgadhandler()
               so we must fall thru */
        } else if( g == &g_back ) {
            if( its ) {
                topic = *(struct TopicStruct *)its;
                req.CurrIt = topic.cursory;
                req.WinTop = topic.windowy;
                its = PopStack( its );
            }
        }
    } while( g && g != &g_ok );

    /* push topic on stack */
    topic.cursory = req.CurrIt;
    topic.windowy = req.WinTop;
    its = PushStack( its, (APTR)&topic, sizeof(topic) );

    SetCursorPos( sav.XPos, sav.YPos );

done:

    if( winopen )
        CloseWindow( &win );

    --recur;
    SetHook( savint, HOOK_WAIT );

    if( fptr != -1 ) Close( fptr );
    if( rtab_first ) free( rtab_first );
    if( tbuf ) {
        free( tbuf );
        tbuf = NULL;
    }

    ResetCount();
}
#undef WIN_WIDTH
#undef WIN_HEIGHT

#define DEFHELPCOL ATTR(CYAN,BRIGHTWHITE)
#define DEFBUTTONCOL ATTR(CYAN,YELLOW)

static struct Gadget *helpgadhandler( struct Gadget *f, struct Gadget *g,
    unsigned left_off, unsigned top_off, unsigned long flags ) {

    struct Request *r;
    int len, numlines, row, col;
    int i;
    char _far *cp;
    BOOL button_pending = FALSE;
    struct InputEvent *ie = IntuitionBase->InputEvent;

    r = (struct Request *)(g->SpecialInfo);

    if( flags & CG_OWNREDRAW ) {
        if( flags & CG_CHANGE ) {
            /* handle mouse and unsupported keyboard event */
            if( ie->buttons & MBUTTON_LEFT ) {
                /* check box dimensions */
                if( ie->mousey > g->Top + top_off
                  && ie->mousey < g->Top + top_off + g->Height - 1
                  && ie->mousex > g->Left + left_off
                  && ie->mousex < g->Left + left_off + g->Width - 2 ) {
                    /* klicked inside box */
                    r->CurrIt = r->WinTop + ie->mousey - top_off - g->Top - 1;
                    topic.cursorx = ie->mousex - left_off - g->Left - 1;
                    /* wait for button release */
                    while( ie->buttons & MBUTTON_LEFT ) ie = GetInput();
                    /* force handling below */
                    ie->key = BK_RETURN;
                    ie->buttons = 0;
                }
            } else if( ie->key ) {
                switch( ie->key ) {
                    case BK_LEFT:
                        if( topic.cursorx ) --topic.cursorx;
                        /* prevent from default behaviour of INTUITION */
                        ie->key = 0;
                        break;
                    case BK_RIGHT:
                        if( topic.cursorx < g->Width-4 ) ++topic.cursorx;
                        ie->key = 0;
                        break;
                    case BK_HOME:
                        r->CurrIt = r->WinTop = 0;
                        ie->key = 0;
                        break;
                    case BK_END:
                        r->CurrIt = r->NumIt-1;
                        r->WinTop = _MAX( (int)(r->NumIt-g->Height+2), 0 );
                        ie->key = 0;
                        break;
                    case BK_TAB:
                        /* skip title and first r->CurrIt lines */
                        for( cp = tbuf, row = -1; row < r->CurrIt; ++row ) {
                            len = *(unsigned _far *)cp;
                            cp += sizeof(unsigned) + len;
                        }
                        len = *(unsigned _far *)cp;
                        cp += sizeof(unsigned);
                        for( col = 0; col <= topic.cursorx && len > 0; ) {
                            switch( *cp ) {
                                case 01: cp += 2; len -= 2;
                                         break;
                                case 02: ++cp; --len;
                                         break;
                                case 03: cp += 3; len -= 3;
                                         break;
                                default: ++cp; --len;
                                         ++col;
                            }
                        }
                        /* now cp points to the current position */
                        /* scan until end of window */
                        numlines = _MIN( r->NumIt, r->WinTop+g->Height-2 );
                        for( ;row < numlines; ++row ) {
                            for( ; len > 0; ) {
                                switch( *cp ) {
                                    case 01: cp += 2; len -= 2;
                                             break;
                                    case 02: /* start button */
                                             r->CurrIt = row;
                                             topic.cursorx = col;
                                             ie->key = 0;
                                             SetCursorPos( g->Left + left_off + col + 1,
                                                           g->Top + top_off + row - r->WinTop + 1 );
                                             return g;
                                    case 03: cp += 3; len -= 3;
                                             break;
                                    default: ++cp; --len;
                                             ++col;
                                }
                            }
                            col = 0;
                            len = *(unsigned _far *)cp;
                            cp += sizeof(unsigned);
                        }
                        return g;
                }
            }
            if( ie->key == BK_RETURN ) {
                /* look for reference */
                /* skip title and first r->CurrIt lines */
                for( cp = tbuf, i = r->CurrIt; i >= 0; --i ) {
                    len = *(unsigned _far *)cp;
                    cp += sizeof(unsigned) + len;
                }
                len = *(unsigned _far *)cp;
                cp += sizeof(unsigned);
                for( i = 0; i <= topic.cursorx; ) {
                    switch( *cp ) {
                        case 01: cp += 2;
                                 break;
                        case 02: ++cp;
                                 button_pending = TRUE;
                                 break;
                        case 03: cp += 3;
                                 button_pending = FALSE;
                                 break;
                        default: ++cp;
                                 ++i;
                    }
                }
                if( button_pending ) {
                    /* inside button */
                    /* push topic on stack */
                    topic.cursory = r->CurrIt;
                    topic.windowy = r->WinTop;
                    its = PushStack( its, (APTR)&topic, sizeof(topic) );
                    topic.cursorx = -1;
                    while( *cp != 03 ) ++cp;
                    ++cp;
                    topic.offset = *(unsigned _far *)cp;
                } else
                    ie->key = 0;
                return g;
            }
        }

        /* redraw box */
        numlines = _MIN( r->NumIt, g->Height-2 );
        /* skip title and first r->WinTop lines */
        for( cp = tbuf, i = r->WinTop ; i >= 0; --i ) {
            len = *(unsigned _far *)cp;
            cp += sizeof(unsigned) + len;
        }

        row = top_off + g->Top + 1;
        col = left_off + g->Left + 1;
        for( i = 0; i < numlines; ++i, ++row ) {
            cp = _printline( cp, row, col, g->Width-3 );

            if( i + r->WinTop == r->CurrIt )
                SetCursorPos( g->Left + left_off + topic.cursorx + 1,
                    g->Top + top_off + i + 1 );

        }
        /* clear rest of window (if any) */
        for( ; i < g->Height-2; ++i, ++row )
            _printline( "\0", row, col, g->Width-3 );
    }
}

static char *_printline( char *cp, unsigned row, unsigned col, int width ) {

    unsigned attr = CR_HELP, oldattr;
    int len;

    len = *(unsigned _far *)cp;
    cp += sizeof(unsigned);

    while( len > 0 ) {
        switch( *cp ) {
            case 01: /* change color */
                ++cp;
                --len;
                attr = (int)*cp++;
                if( attr == 0 ) attr = CR_HELP;
                --len;
                break;
            case 02: /* start button */
                ++cp;
                --len;
                oldattr = attr;
                attr = CR_HELPBUTTON;
                break;
            case 03: /* end button */
                cp += 3;
                len -= 3;
                attr = oldattr;
                break;
            default: /* any char */
                if( width > 0 )
                    VideoOut( attr, *cp, col, row );
                ++cp;
                --len;
                ++col;
                --width;
        }
    }
    while( width > 0 ) {
        VideoOut( attr, ' ', col, row );
        ++col;
        --width;
    }

    return cp;
}

/* end of file help.c */
