# -*-perl-*-
# parsepl.pl
# (C) A. J. C. Duggan 22/9/93
# Parsing package for TeX's PL and VPL files
#
# public routines are:
#	number(property)	Returns numeric value of number property
#	fontname(file)		Returns default font name for filename
#	plactions(name, act...)	Register property actions for parsing
#	parsepl(file)		Open and parse PL/VPL file using plactions
#	expand(props)		Output indented property list
# default plactions provided are:
#	ignore(props)		Ignore property or property list
#	property(prop)		Generate (property) form
#	list(proplist)		Generate (proplist) form

###############################################################################
# Property list manipulation routines
###############################################################################

package parsepl;

# number(property)
# returns the value of number defined by property
sub main'number {
   local($_, $value) = split(' ', shift);
   local(%xerox) = (MRR, 0, MIR, 1, BRR, 2, BIR, 3, LRR, 4, LIR, 5, MRC, 6,
		    MIC, 7, BRC, 8, BIC, 9, LRC, 10, LIC, 11, MRE, 12, 
		    MIE, 13, BRE, 14, BIE, 15, LRE, 16, LIE, 17);
 NUMBER:
   {
      /^C$/ && (($value) = unpack(C, $value), last NUMBER);
      /^D$/ && ($value = $value+0, last NUMBER);
      /^F$/ && ($value = $xerox{$value}, last NUMBER);
      /^O$/ && ($value = oct($value), last NUMBER);
      /^H$/ && ($value = hex($value), last NUMBER);
      /^R$/ && ($value = $value+0.0, last NUMBER);
      &main'fatal("bad number: $_ $value");
   }
   $value;
}

# fontname(file)
# return default fontname of file
sub main'fontname {
   local($_) = shift;
   print STDERR "Fontname for $_ is " if $main'debug;
   s:^.*/::;			# drop directory names
   s/\.[^\.]*$//;		# drop extension
   print STDERR "$_\n" if $main'debug;
   $_;
}

# plactions(propname, action, ...)
# register property list actions
sub main'plactions {
   local($propname, $action);
   local($package) = caller;
   while (@_) {
      ($propname, $action) = (shift, shift);
      $actions{$propname} = "${package}'$action";
   }
}

# PL property functions
# called by parsepl with depth and normal arguments as parameters
sub main'ignore {		# ignore
   print STDERR join(' ', "Ignoring", @_), "\n" if $main'debug;
   undef;
}

sub main'property {		# return as property
   '('.join(' ', @_).')';
}

sub main'list {			# return as property list
   local($name) = shift;
   join("\n", "($name", @_, ')');
}

# list of property -> parameters
#    (N=number, S=string, P=property list, L=label or number)
%parameters = (CHECKSUM, S, DESIGNSIZE, N, DESIGNUNITS, N, CODINGSCHEME, S,
	       FAMILY, S, FACE, N, SEVENBITSAFEFLAG, S, HEADER, NN,
	       BOUNDARYCHAR, N, VTITLE, S, COMMENT, S,
	       FONTDIMEN, P,	# FONTDIMEN properties follow
	       SLANT, N, SPACE, N, STRETCH, N, SHRINK, N, XHEIGHT, N, QUAD, N,
	       EXTRASPACE, N, NUM1, N, NUM2, N, NUM3, N, DENOM1, N, DENOM2, N,
	       SUP1, N, SUP2, N, SUP3, N, SUB1, N, SUB2, N, SUPDROP, N,
	       SUBDROP, N, DELIM1, N, DELIM2, N, AXISHEIGHT, N,
	       DEFAULTRULETHICKNESS, N, BIGOPSPACING1, N, BIGOPSPACING2, N,
	       BIGOPSPACING3, N, BIGOPSPACING4, N, BIGOPSPACING5, N,
	       PARAMETER, NN,
	       LIGTABLE, P, # LIGTABLE properties follow
	       LABEL, L, KRN, NN, STOP, '', SKIP, N, LIG, NN, '/LIG', NN,
	       '/LIG>', NN, 'LIG/', NN, 'LIG/>', NN, '/LIG/', NN, '/LIG/>', NN,
	       '/LIG/>>', NN,
	       MAPFONT, NP, # MAPFONT properties follow
	       FONTDSIZE, N, FONTNAME, S, FONTAREA, S, FONTCHECKSUM, N,
	       FONTAT, N,
	       CHARACTER, NP,	# CHARACTER properties follow
	       CHARWD, N, CHARHT, N, CHARDP, N, CHARIC, N, NEXTLARGER, N,
	       VARCHAR, P,	# VARCHAR properties follow
	       TOP, N, MID, N, BOT, N, REP, N,
	       MAP, P,	# MAP properties follow
	       SELECTFONT, N, SETCHAR, N, SETRULE, NN, PUSH, '', POP, '',
	       MOVERIGHT, N, MOVELEFT, N, MOVEUP, N, MOVEDOWN, N, SPECIAL, S,
	       SPECIALHEX, S
	       );

@tokens = ();			# list of tokens still to be processed
$token = undef;			# current token value

# gettoken(notblank, [notrequired])
# gets the next non-null token from the input file, causes an error if there
# are no tokens and notrequired is false (or not present). Uses the file name
# from ancestor functions.
sub gettoken {
   local($blankok, $notrequired) = @_;
   for (;;) {
      while (@tokens) {
	 $token = shift(@tokens);
	 return $token if $token ne '' && ($blankok || $token !~/^\s*$/);
      }
      if ($_ = <$file>) {
	 @tokens = split(/(\(|\)|\s+)/);
      } else {
	 return $token = undef if $notrequired;
	 &main'fatal("property list $file ended early");
      }
   }
}

# getstring()
# returns a token representing a string
sub getstring {
   local($string, $blanks, $paren);
   while (&gettoken($blanks++) ne ')' || $paren) {
      $string .= $token;
      $paren++ if $token eq '(';
      $paren-- if $token eq ')';
   }
   unshift(@tokens, $token);
   $string;
}

# getnumber(sep)
# returns a list representing a number
sub getnumber {
   join(' ', &gettoken(), &gettoken());
}

# getproperty(separator, outer)
# returns a property, with the parts separated by the separator
# outer determines if it is an outer-level property
sub getproperty {
   local($outer) = shift;
   local(@property, $propname, $fn);
   if (&gettoken(0, $outer) ne undef) {
      if ($token eq '(') {
	 print STDERR '.' if !$main'debug && !$main'quiet;
	 push(@property, $propname = &gettoken()); # get property name
	 &main'fatal("unknown property name $propname in file $file")
	    if !defined($parameters{$propname});
         $fn = $actions{$propname};
	 foreach (split(//, $parameters{$propname})) {
	    if (/N/) {		# number required
	       push(@property, &getnumber());
	    } elsif (/S/) {	# string required
	       push(@property, &getstring());
	    } elsif (/L/) {	# label or number required
	       if (&gettoken() eq BOUNDARYCHAR) {
		  push(@property, $token);
	       } else {
		  unshift(@tokens, $token);
		  push(@property, &getnumber());
	       }
	    } elsif (/P/) {	# property list required
	       push(@property, &getproplist());
	    } else {		# internal table error
	       &main'fatal("this can't happen; property $propname parameter is $_");
	    }
	 }
	 &main'fatal("$propname parameter list terminated by $token in $file")
	    if &gettoken() ne ')';
	 $property = do $fn(@property)
	    if $fn ne undef;
      } else {			# token wasn't (, so restore it.
	 unshift(@tokens, $token);
	 $property = EOL;
      }
   } else {
      $property = EOF;
   }
   $property;
}

# getproplist(separator, outer)
# returns a property list, with the properties separated by the separator
# outer determines if it is an outer-level property
sub getproplist {
   local(@proplist);
   for (;;) {
      &getproperty(0);
      last if $property eq EOL;
      push(@proplist, $property);
   }
   @proplist;
}

# parsepl(file)
# parses the property list in file
sub main'parsepl {
   local($file) = shift;
   local($caller) = caller;
   foreach ("designsize = 10", "designunits = 1", "codingscheme = UNSPECIFIED",
	    "family = UNSPECIFIED") {
      eval "\$${caller}'$_";
   }
   print STDERR "Looking for PL/VPL $file\n" if !$main'quiet; #'
   if (open($file, $file) || open($file, "$file.pl") ||
       open($file, "$file.vpl")) {
      while (&getproperty(1) ne EOF) {
	 print STDERR '/' if !$main'debug && !$main'quiet;
      };
      &main'fatal("property list file $file contains extra tokens $token...")
	 if $token ne undef;
      print STDERR "\n" if !$main'debug && !$main'quiet;
      close($file);
   } else {
      &main'fatal("can't find PL or VPL file for $file");
   }
}

###############################################################################
# VPL output routines
###############################################################################

# expand(pl)
# print balanced property list with proper indentation
sub main'expand {
   local($indent) = 0;
   local(@list) = @_;
   foreach (@list) {
      foreach (split("\n")) {
	 s/^\s*//;
	 next if /^$/;
	 print '   ' x $indent, $_, "\n";
	 $indent += split('\(', $_, -1) - split('\)', $_, -1);
      }
   }
}

1;
