/* options.c - 19:28 GMT +10:00 Fri 20 Aug 1993 - modifier Geoffrey Tobin */

/* From input file "../include/globals.p" */

#define CHECK_ON_DVIname
#define EXIT_IF_NO_DVI
#define NEW_DIMEN_PIXELS

#include "config.h"
#include "globals.h"
#include "screenio.h"
#include "options.h"
#include "defaults.h"


/* Variables Exported, via "options.h" */

/* X and Y resolutions, in dpi */
double xres, yres;

/* Magnification */
int mag;

/* The next four are in paper pixels, and calculated. */
int hoffset, voffset;
int paperwd, paperht;

string dummy_pk, dummy_tfm;
string pkdir, tfmdir;
string psprefix;
string cmd_help, immed_help;
string vdu;

/* DVI file's name */
string DVIname;


typedef struct
{
    units unit;
    char * name;
    double per_inch;
} Dan;  /* `Dimension and name' */

static
Dan dans[] =  /* `dimensions and names series' */
{
    {ic, "in", 1.0},
    {cm, "cm", 2.54},
    {mm, "mm", 25.4},
    {bp, "bp", 72.0},
    {pc, "pc", 72.27 / 12},
    {pt, "pt", 72.27},
    {sp, "sp", 72.27 * 65536},
    {px, "px", 300.0}   /* "px"'s per_inch value is not used, */
};                      /* but Qume, Laserwriter, etc, are 300 dpi */

static
size_t ldans = sizeof (dans) / sizeof (Dan);  /* length of dans[] */

Static int Inint __((double f));

Static string clarg;  /* current command line argument */
Static int clen;      /* command line argument's length */
Static char option;   /* current command option */
Static string value;  /* current option's value */
Static int vlength;   /* current value's length */
Static int argnum;   /* 0..argc-1; used in argv calls */


/******************************************************************************/

/* Auxiliary routines */

/******************************************************************************/

/* Copy at most n characters from ct to s; set s[n] = NUL character. */
/* This ASSUMES that s has memory >= n+1 characters. */

#ifdef __STDC__
char * strMcpy (char * s, const char * ct, size_t n)
#else
char * strMcpy (s, ct, n)
  char * s;
  char * ct;
  size_t  n;
#endif
{
  strncpy (s, ct, n);
  s[n] = '\0';
  return s;
}
/* strMcpy */

/******************************************************************************/

/* Copy at most maxstring characters from ct to s; */
/* set s[maxstring] = NUL character. */

#ifdef __STDC__
char * stringcopy (string s, const char * ct)
#else
char * stringcopy (s, ct)
  string s;
  char * ct;
#endif
{
  strncpy (s, ct, maxstring);
  s[maxstring] = '\0';
  return (char *) &s;
}
/* stringcopy */

/******************************************************************************/

#ifdef __STDC__
Static boolean HasExplicitExt (char * fname, int len)
#else
Static boolean HasExplicitExt (fname, len)
    char * fname;
    int len;
#endif
{
  /* SYSDEP: Check for an extension of the form ".*" where * is any string
     not containing "/".  len is length of fname.
     If "." found, then TRUE is returned, otherwise FALSE.
     - gt - no side effects.
  */

  boolean Result = false;

  while (len > 0)  /* search backwards looking for . */
  {
    len--;
    if (fname[len] == '/')  /* handle file names like "../myfiles/foo" */
    {
      Result = false;
      break;
    }
    if (fname[len] == '.')
    {
      Result = true;
      break;
    }
  }
  return Result;
}
/* HasExplicitExt */

/******************************************************************************/

Static Void GetValue (VOID)
{
/* gt observes - global variables:
 *  read (I):   written (O):
 *
 .    P_argc      -      .
 .    argnum    argnum   .
 .    option      -      .
 .    clarg       -      .
 .    clen        -      .
 .       -      value    .
 .       -      vlength  .
 */

  /* Get wanted parameter following current option and store in value. */

  if (clen > 2)  /* allow things like -m1000 */
  {
    /* This would be easier if value were a variable pointer.    */
    /* Then we could just say "value = clen + 2;" instead of the loop. */
    int i;
    for (i = 0; i < clen; i++)   /* shift value left 2 places */
      value[i] = clarg[i + 2];
    vlength = clen - 2;
  }
  else  /* clen <= 2, and current option should be followed by _one_ value */
  {
    if (argnum < P_argc)  /* More command line arguments */
    {
      stringcopy (value, P_argv[argnum]);

      /* gt - this needs to be here in the if statement. */
      /*    - otherwise, value = -option, which isn't right. */
      /*    - also, increasing argnum wd otherwise be wrong. */

      vlength = strlen (value);   /* 0 if no more args */
      argnum++;
    }
    else  /* gt - missing value */
    {
      stringcopy  (value, "");
      vlength = 0;
    }

    if (vlength < 0)
    {
      MesgString ("options.c GetValue(): ");
      MesgString ("This can't happen! ");
      MesgString ("After -");
      MesgChar (option);
      MesgString (", vlength < 0 !");
      MesgLine ();

      RestoreTerminal ();
      exit (1);
    }
#ifdef OLD_FUSSY_CHECK
    else if (vlength == 0)
    {
      MesgString ("Missing value after -");
      MesgChar (option);
      MesgLine ();
/*
      RestoreTerminal ();
      exit (1);
*/
    }
#endif  /* OLD_FUSSY_CHECK */
  }
}
/* GetValue */

/******************************************************************************/

#ifdef __STDC__
Static int Inint (double f) 
#else
Static int Inint (f) 
    double f;
#endif
{
  return ((int) (f + 0.5));
}
/* Inint */

/******************************************************************************/

#ifdef __STDC__
Static Void ToCardinal (const char * option, const char * value, int *n)
#else
Static Void ToCardinal (option, value, n)
    char * option;  char * value;  int * n;
#endif
{
  /* If value represents a positive integer, then return via n. */

  int check;
  double r;

  int vlen;
  char * sval;

  vlen = strlen (value);
  sval = (char *) Malloc ((size_t) (vlen+1));
  (void) strcpy (sval, value);

  check = sscanf (sval, "%lf", &r);

  Free (sval, char);

  *n = Inint (r);  /* round value to nearest integer */

  /* check can be 1 or EOF if ok! */
  if ((check == 1 || check == EOF) && *n > 0)
    return;

  MesgString ("Bad");
  MesgString (option);
  MesgString (" value: ");
  MesgString (value);
  MesgLine();

  MesgString ("Specify a positive integer.");
  MesgLine();

  RestoreTerminal();
  exit (1);
}
/* ToCardinal */

/******************************************************************************/

#ifdef __STDC__
Static Void ToRes (const char * option, const char * value, double * xr, double * yr)
#else
Static Void ToRes (option, value, xr, yr)
    char * option;  char * value;  double * xr, * yr;
#endif
{
  /* X and Y resolutions, specified in dots per inch. */

  int check;

  int vlen;
  char * sval;

  vlen = strlen (value);
  sval = (char *) Malloc ((size_t) (vlen+1));
  (void) strcpy (sval, value);

  check = sscanf (sval, "%lf,%lf", xr, yr);

  Free (sval, char);

  if (check == 2 && *xr > 0 && *yr > 0)
  {
    return;
  }
  else
  {
    sval = (char *) Malloc ((size_t) (vlen+1));
    (void) strcpy (sval, value);

    check = sscanf (sval, "%lf", xr);

    Free (sval, char);

    if (check == 1 && *xr > 0)
    {
      *yr = *xr;
      return;
    }
  }

  MesgString ("Bad");
  MesgString (option);
  MesgString (" value(s): ");
  MesgString (value);
  MesgLine();

  MesgString ("Resolutions must be positive reals,");
  MesgLine();
  MesgString ("specified as `-r xres' or `-r xres,yres'");
  MesgLine();
  MesgString ("in dots per inch.");
  MesgLine();

  RestoreTerminal();
  exit (1);
}
/* ToRes */

/******************************************************************************/

#ifdef __STDC__
Static units Name2Unit (const char * uname)
#else
Static units Name2Unit (uname)
  char * uname;
#endif
{
  units unit = ic;
  char Luname[3];  /* Lower case unit name */
  int j;

#define ToLower(x)  (isupper(x) ? tolower(x) : (x))

  Luname[0] = ToLower (uname[0]);
  Luname[1] = ToLower (uname[1]);
  Luname[2] = '\0';

  for (j=0; j < ldans && strcmp (Luname, dans[j].name); j++)
    ;   /* Search for the name of the unit. */

  if (j < ldans)
    unit = dans[j].unit;
  else
  {
    MesgString ("Bad unit name `");
    MesgString (uname);
    MesgString ("'");
    MesgLine();

    MesgString ("Last two letters should be one of  ");
    for (j=0; j < ldans; j++)
    {
      MesgString (" `");
      MesgString (dans[j].name);
      MesgString ("'");
    }
    MesgChar ('.');
    MesgLine();

    MesgString ("Using inch.");
    MesgLine();
    unit = ic;
  }
  return unit;
}
/* Name2Unit */

/******************************************************************************/

#ifdef __STDC__
Static Void ToDimen (const char * option, const char * value, double *r, units *un)
#else
Static Void ToDimen (option, value, r, un)
    char * option;  char * value;  double * r;  units * un;
#endif
{
  /* A valid dimension consists of an integer or real number followed
     by a two-letter unit:  see  dans[]  above.
     If value represents a valid dimension,
     return number part in r, and units part in un.
  */

  int check;
  string nun;  /* name of unit - ample space in a string for it */

  int vlen;
  char * sval;

  vlen = strlen (value);
  sval = (char *) Malloc ((size_t) (vlen+1));
  (void) strcpy (sval, value);

  check = sscanf (sval, "%lf%2s", r, nun);

  Free (sval, char);

  if (check == 2)
  {
    /* extract unit's code from its name */
    *un = Name2Unit (nun);
  }
  else if (check == 1)
  {
    MesgString ("No dimension given, using inch.");
    MesgLine();
    *un = ic;
  }
  else  /* check < 1 */
  {
    MesgString ("Bad");
    MesgString (option);
    MesgString (" value: ");
    MesgString (value);
    MesgLine();

    MesgString ("Specify a dimensioned quantity.");
    MesgLine();

    RestoreTerminal();
    exit (1);
  }
}
/* ToDimen */

/******************************************************************************/

#ifdef __STDC__
Static Void ToPosDimen (const char * option, const char * value, double *r, units *un)
#else
Static Void ToPosDimen (option, value, r, un)
    char * option;  char * value;  double * r;  units * un;
#endif
{
  /* A valid +ve dimension consists of a positive integer or real number
     followed by a two-letter unit:  see  dans[]  above.
     If value represents a valid dimension,
     return numeric part in r, and units part in un.
  */

  ToDimen (option, value, r, un);

  if (*r <= 0)
  {
    MesgString ("Bad");
    MesgString (option);
    MesgString (" value: ");
    MesgString (value);
    MesgLine();

    MesgString ("Specify a positive dimension.");
    MesgLine();

    RestoreTerminal();
    exit (1);
  }
}
/* ToPosDimen */

/******************************************************************************/

#ifdef __STDC__
Static int DimenPixels (double r, units u, double res)
#else
Static int DimenPixels (r, u, res)
    double r;  units u;  double res;
#endif
{
  /* Return given dimension as nearest whole number of pixels, */
  /* given the appropriate (eg, X or Y direction's) resolution in dpi. */

  int Result;

#ifdef  NEW_DIMEN_PIXELS

  int i;

  if (u == px)
    Result = Inint (r);
  else
  {
    for (i=0; i < ldans && u != dans[i].unit; i++)
      ;  /* search for unit */

    if (i < ldans)
      Result = Inint (r / dans[i].per_inch * res);
    else
    {
      MesgString ("Unimplemented unit, treating as inch.");
      MesgLine();
      Result = Inint (r * res);
    }
  }

#else

  switch (u)
  {
    case px:
      Result = Inint (r);
      break;
  
    case ic:
      Result = Inint (r * res);
      break;
  
    case sp:
      Result = Inint (r / 72.27 / 65536.0 * res);
      break;

    case cm:
      Result = Inint (r / 2.54 * res);
      break;
  
    case mm:
      Result = Inint (r / 25.4 * res);
      break;
  
    case bp:
      Result = Inint (r / 72.0 * res);
      break;
  
    case pt:
      Result = Inint (r / 72.27 * res);
      break;
  
    case pc:
      Result = Inint (r / 72.27 * 12.0 * res);
      break;

    default:
      MesgString ("Unimplemented unit, treating as inch.");
      MesgLine();
      Result = Inint (r * res);
      break;
  }
#endif  /* NEW_DIMEN_PIXELS */

  return Result;
}
/* DimenPixels */

/******************************************************************************/

#ifdef __STDC__
Void WriteCardinal (char * name, int value)
#else
Void WriteCardinal (name, value)
  char * name;
  int value;
#endif
{
  string outstring;

  sprintf (outstring, "%s= %d", name, value);
  MesgString (outstring);
  MesgLine();
}
/* WriteCardinal */

/******************************************************************************/

#ifdef __STDC__
Void WriteReal (char * name, double value)
#else
Void WriteReal (name, value)
  char * name;
  double value;
#endif
{
  string outstring;

  sprintf (outstring, "%s= %f", name, value);
  MesgString (outstring);
  MesgLine();
}
/* WriteReal */

/******************************************************************************/

/* GT - TO BE FINISHED - Write... FUNCTIONS BELOW, TOO! */
/* Might be better to use a pair of arrays, or array of pairs */
/* That is, data-driven may be more flexible. */

#ifdef __STDC__
Void NameUnit (units un, string nun)
#else
Void NameUnit (un, nun)
  units un;
  string nun;
#endif
/* Interface:  un (I),  *nun (O). */
{
  int i;

  for (i=0; i < ldans && un != dans[i].unit; i++)
    ;  /* Search for unit un. */

  if (i < ldans)
    stringcopy (nun, dans[i].name);
  else
  {
      MesgString ("Unknown unit!  Assuming inch.");
      MesgLine();

      stringcopy (nun, "in");
  }
}
/* NameUnit */

/******************************************************************************/

#ifdef __STDC__
Void WriteDimen (char * varname, double r, units un)
#else
Void WriteDimen (varname, r, un)
  char * varname;
  double r;
  units un;
#endif
{
  string outstring, nun;

  NameUnit (un, nun);
  /* !! gt - this line is unsafe, as outstring may overflow! */
  sprintf (outstring, "%s= %f %s", varname, r, nun);
  MesgString (outstring);
  MesgLine();
}
/* WriteDimen */

/******************************************************************************/

#ifdef __STDC__
Void WriteName (char * varname, char * svalue)
#else
Void WriteName (varname, svalue)
  char * varname;
  char * svalue;
#endif
{
  string outstring;

  sprintf (outstring, "%s= %s", varname, svalue);
  MesgString (outstring);
  MesgLine();
}
/* WriteName */

/******************************************************************************/

#ifdef __STDC__
double Unit2Value (units unit)
#else
double Unit2Value (unit)
  units unit;
#endif
{
  double per_inch = 1.0;  /* 1 unit per inch */
  int j;

  for (j=0; (j < ldans) && (unit != dans[j].unit); j++)
    ;   /* Search for the unit. */

  if (j < ldans)
    per_inch = dans[j].per_inch;
  else
  {
    string ustr;
    sprintf (ustr, "%d", unit);
    MesgString ("Bad unit, enumeration value = ");
    MesgString (ustr);
    MesgLine();

    RestoreTerminal();
    exit (1);
  }
  return per_inch;
}
/* Unit2Value */

/******************************************************************************/

Void InitOptions (VOID)
{
  /* Get DVI file, and any options, from the command line.
     If an option appears more than once, then we return the last value.

     Actually, proceed in three steps, in increasing priority:
     (1)  Initialise to default values, as set in "defaults.h".
     (2)  If the appropriate environment variable is set, use that.
     (3)  If the appropriate command line option is set, use that.
  */

  char * env;    /* environment variable */
  units def_unit = ic;  /* default unit is inch */
  units hoffu, voffu, xu, yu;
  double hoffr, voffr, xr, yr;
  boolean landscape;

  /*
     (1)  DEFAULTS.
     initialize option values with defaults; note that the dv script can
     set up different defaults  [- quoth Andrew Trevorrow, I presume]
     - gt observes - no "dv" script provided with DVItoVDU version 3.0 !
  */
  
  /* "dpi" means "dots per inch" */
  /* printer's X and Y resolutions in dpi */
  xres = (double) DEF_XRES;
  yres = (double) DEF_YRES;

  /* Page Offsets from Paper edges */
  hoffu = def_unit;
  voffu = def_unit;
  hoffr = 0.0;   /* no margin shifting, so 1" left and top margins */
  voffr = 0.0;

  mag = 0;       /* use DVI file's intrinsic magnification */

  /*
      (2)  ENVIRONMENT VARIABLES.
  */

  /* Resolution, in dots per inch */
  /* If DV_RES is set in the environment, then try to use that. */
  /* If two real values are in DV_RES, copy them to xres and yres. */
  /* If only one is in DV_RES, copy it to both xres and yres. */
  /* If none is, then ignore DV_RES. */

  if ((env = getenv ("DV_RES")) != (char *) NULL)
  {
    if (sscanf (env, "%lf%lf", &xres, &yres) == 1)
      yres = xres;
  }

  /* Display type:  default is SYSDEP; see the dv script */
  /* gt - no "dv" script provided with version 3.0, sadly for us! */

  if (!(env = getenv ("DV_TERM")))
    env = getenv ("TERM");

  stringcopy (vdu,
    !env ? "" : env);

  /* Paper Width */
  ToPosDimen ("", !(env = getenv ("DV_PAPERWD")) ? DEF_PAPERWD : env,
              &xr, &xu);

  /* Paper Height */
  ToPosDimen ("", !(env = getenv ("DV_PAPERHT")) ? DEF_PAPERHT : env,
              &yr, &yu);

  /* location of PK files */

  if (!(env = getenv ("TEXPKS")) && !(env = getenv ("PKFONTS")))
    env = getenv ("TEXFONTS");

  stringcopy (pkdir,
    !env ? DEF_PK_DIR : env);

  /* location of TFM files */

  stringcopy (tfmdir,
    ! (env = getenv ("TEXFONTS")) ? DEF_TFM_DIR : env);

  /* fall-back PK font */

  stringcopy (dummy_pk,
    ! (env = getenv ("DV_DUMMY_PK")) ? DEF_DUMMY_PK : env);

  /* fall-back TFM metric */

  stringcopy (dummy_tfm,
    ! (env = getenv ("DV_DUMMY_TFM")) ? DEF_DUMMY_TFM : env);

  /* prefix in PS font names */

  stringcopy (psprefix,
    ! (env = getenv("DV_PS_PREFIX")) ? DEF_PS_PREFIX : env);

  /* command help file read by ? command */

  stringcopy (cmd_help,
    ! (env = getenv ("DV_CMD_HELP")) ? DEF_CMD_HELP : env);

  /* immediate help file read when dvgt has no arguments */

  stringcopy (immed_help,
    ! (env = getenv ("DV_IMMED_HELP")) ? DEF_IMMED_HELP : env);

  /* DVI file name */

  stringcopy (DVIname, "");  /* no DVI file yet */

  /* portrait or landscape page dimensions? */

  landscape = false;   /* portrait:  don't swap -x and -y values */

  /*
      (2)  COMMAND LINE OPTIONS.
  */

  argnum = 1;
  while (argnum < P_argc)
  {
    stringcopy (clarg, P_argv[argnum]);
    clen = strlen (clarg);
    argnum++;

    if (clarg[0] == '-')
    {
      if (clen < 2)
      {
        if (clen < 1)
        {
          MesgString ("This cannot happen!  argument '-' with length < 1");
          MesgLine();
        }
        else if (clen == 1)
        {
          MesgString ("Valid options are:");
          MesgLine();
          MesgString ("  H V d e f h i l m p r t v x y");
          MesgLine();
        }
        RestoreTerminal();
        exit (1);
      }
      else
        option = clarg[1];

      /* bad string values will be detected in other modules */

      switch (option)
      {

        case 'H':
          GetValue();
          if (vlength > 0)
            ToDimen (" -H", value, &hoffr, &hoffu);
          else
            WriteDimen ("horizontal offset", hoffr, hoffu);
          break;

        case 'V':
          GetValue();
          if (vlength > 0)
            ToDimen (" -V", value, &voffr, &voffu);
          else
            WriteDimen ("vertical offset", voffr, voffu);
          break;

        case 'd':
          GetValue();
          if (vlength > 0)
            stringcopy (dummy_pk, value);
          else
          {
            WriteName ("dummy PK font", dummy_pk);
            MesgString ("environment variable = DV_DUMMY_PK");
            MesgLine();
          }
          break;

        case 'e':  /* gt - not very mnemonic, I know */
          GetValue();
          if (vlength > 0)
            stringcopy (dummy_tfm, value);
          else
          {
            WriteName ("dummy TFM metric", dummy_tfm);
            MesgString ("environment variable = DV_DUMMY_TFM");
            MesgLine();
          }
          break;

        case 'f':
          GetValue();
          if (vlength > 0)
            stringcopy (pkdir, value);
          else
          {
            WriteName ("pk font directory", pkdir);
            MesgString ("environment variable = TEXPKS or PKFONTS or TEXFONTS");
            MesgLine();
          }
          break;

        case 'h':
          GetValue();
          if (vlength > 0)
            stringcopy (cmd_help, value);
          else
          {
            WriteName ("command help file", cmd_help);
            MesgString ("environment variable = DV_CMD_HELP");
            MesgLine();
          }
          break;

        case 'i':
          WriteName ("immediate help file", immed_help);
          MesgString ("environment variable = DV_IMMED_HELP");
          MesgLine();
          break;

        case 'l':
          landscape = true;
          MesgString ("In landscape orientation now.  (Default is portrait.)");
          MesgLine();
          break;

        case 'm':
          GetValue();
          if (vlength > 0)
            ToCardinal (" -m", value, &mag);
          else
            WriteCardinal ("magnification", mag);
          break;

        case 'p':
          GetValue();
          if (vlength > 0)
            stringcopy (psprefix, value);
          else
          {
            WriteName ("postscript font prefix", psprefix);
            MesgString ("environment variable = DV_PS_PREFIX");
            MesgLine();
          }
          break;

        case 'r':
          GetValue();
          /* gt - resolutions should allow non-integral dpi */
          /*    - this is because there are such output devices */
          /*    - for example, metric printers.                 */
          if (vlength > 0)
	  {
	    /* Get resolutions, specified in dots per inch */
	    ToRes (" -r", value, &xres, &yres);
	  }
          else
          {
            WriteReal ("horizontal resolution (in dpi) ", xres);
            WriteReal ("vertical   resolution (in dpi) ", yres);
            MesgString ("environment variable = DV_RES");
            MesgLine();
          }
          break;

        case 't':
          GetValue();
          if (vlength > 0)
            stringcopy (tfmdir, value);
          else
          {
            WriteName ("tfm directory", tfmdir);
            MesgString ("environment variable = TEXFONTS");
            MesgLine();
          }
          break;

        case 'v':
          GetValue();
          if (vlength > 0)
            stringcopy (vdu, value);
          else
          {
            WriteName ("vdu", vdu);
            MesgString ("environment variable = DV_TERM or TERM");
            MesgLine();
          }
          break;

        case 'x':
          GetValue();
          if (vlength > 0)
            ToPosDimen (" -x", value, &xr, &xu);
          else
	  {
            WriteDimen ("page width", xr, xu);
            MesgString ("environment variable = DV_PAPERWD");
            MesgLine();
	  }
          break;

        case 'y':
          GetValue();
          if (vlength > 0)
            ToPosDimen (" -y", value, &yr, &yu);
          else
          {
            WriteDimen ("page height", yr, yu);
            MesgString ("environment variable = DV_PAPERHT");
            MesgLine();
          }
          break;

        default:
          MesgString ("Unknown option: `-");
          MesgChar (option);
          MesgString ("'");
          MesgLine();

          MesgString ("Valid options are:");
          MesgLine();
          MesgString ("  H V d e f h i l m p r t v x y");
          MesgLine();

          RestoreTerminal();
          exit (1);

          break;
      }  /* switch */
    }
    else
    /* clarg doesn't start with '-', so assume it's a DVI file's name */
    {
      stringcopy (DVIname, clarg);

      /* bad DVIname will be detected upon open in main module */

      if (HasExplicitExt (DVIname, clen))
        ;    /* dvi file name has an extension */
      else if (clen + 4 <= maxstring)   /* room to append ".dvi" */
      {
        DVIname[clen] = '.';
        DVIname[clen + 1] = 'd';
        DVIname[clen + 2] = 'v';
        DVIname[clen + 3] = 'i';
        DVIname[clen + 4] = '\0';
      }
      else   /* user has given a mighty long file name */
      {
        MesgString ("DVI file name too long: ");
        MesgString (DVIname);
        MesgLine();

        RestoreTerminal();
        exit (1);   /* gt - pretty sad! */
      }  /* if */
    }  /* if */
  }  /* while */

#ifdef CHECK_ON_DVIname
  if (DVIname == (char *) NULL || strlen (DVIname) == 0 || DVIname[0] == ' ')
  {
    /* no file name on command line */

    MesgString ("DVI file not given.");
    MesgLine();
#ifdef EXIT_IF_NO_DVI
    RestoreTerminal();
    exit (1);
#endif /* EXIT_IF_NO_DVI */
  }
#endif /* CHECK_ON_DVIname */

  /* set h/voffset and paperwd/ht only after resolutions have been decided */
  hoffset = DimenPixels (hoffr, hoffu, xres);
  voffset = DimenPixels (voffr, voffu, yres);

  paperwd = DimenPixels (xr, xu, xres);
  paperht = DimenPixels (yr, yu, yres);

  if (landscape)
  {  /* swap paperwd and paperht */
    int temp = paperwd;
    paperwd = paperht;
    paperht = temp;
  }
}
/* InitOptions */

/******************************************************************************/

/* end options.c */
